# run_skeleton_task.py (Completed Skeleton for Predicate Exploration and Plan Execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === STEP 1: Exploration for Predicate Discovery ===
        print("[Exploration] Attempting to identify missing predicates using exploration actions...")

        # Since PDDL feedback suggests a missing predicate (likely around lock, id, weight, temperature etc.),
        # we perform available exploration actions (if implemented in skill_code) corresponding to the 'exploration knowledge'.
        # We'll try to execute each action safely and capture issues.

        exploration_skills = [
            'execute_go_identify',         # discover if objects at location are identified
            'execute_go_temperature',      # discover temperatures
            'execute_pick_weight',         # discover weight
            'execute_pick_durability',     # discover durability
            'execute_pull'                 # discover lock-known (domain-specific, already present in primitive skills)
        ]

        # We'll attempt to invoke these skills (if implemented in skill_code) for the first available object/location found.
        found_obj = None
        found_loc = None
        found_robot = None
        try:
            # Simple extraction of first object and location, and robot if exists
            for k, v in positions.items():
                if 'robot' in k or 'Robot' in k:
                    found_robot = k
                elif 'drawer' not in k and 'handle' not in k:
                    if not found_obj:
                        found_obj = k
                if 'drawer' not in k and 'handle' not in k:
                    if not found_loc:
                        found_loc = v
            # Fallback if above logic fails
            if not found_robot:
                found_robot = 'robot'
        except Exception as e:
            print("[Exploration] Could not determine robot/object/location names from positions:", e)

        # Now, try available exploration skills (these might not all be implemented; if not, skip gracefully)
        for skill in exploration_skills:
            fn = globals().get(skill, None)
            if fn is not None:
                try:
                    print(f"[Exploration] Attempting {skill} ...")
                    if skill == 'execute_go_identify' or skill == 'execute_go_temperature':
                        # Simulate going from one place to another for identify/temperature
                        obs, reward, done = fn(env, task, found_robot, found_loc, found_loc)
                    elif skill == 'execute_pick_weight' or skill == 'execute_pick_durability':
                        obs, reward, done = fn(env, task, found_robot, found_obj, found_loc)
                    elif skill == 'execute_pull':
                        obs, reward, done = fn(env, task, found_robot, found_obj, found_loc)
                    print(f"[Exploration] {skill} executed. (done: {done})")
                except Exception as e:
                    print(f"[Exploration] {skill} failed: {e}")
            else:
                print(f"[Exploration] {skill} is not implemented in skill_code, skipping.")

        print("[Exploration] Exploration phase complete.\n")

        # === STEP 2: Plan Execution ===
        print("[Plan] Executing oracle plan using available skills.")

        # Example plan steps (the actual logic can be replaced with oracle plan steps as needed)
        # For demonstration, let's try a sequence based on the domain:
        # 1. execute_go: move robot to object's location
        # 2. execute_pick: pick up the object from floor
        # 3. execute_go: move to drawer
        # 4. execute_pull: open the drawer by pulling handle
        # 5. execute_place: put the object inside the drawer
        # 6. execute_push: close the drawer

        # We'll use whatever keys are available; adapt for your environment/plan.
        object_key, drawer_key, handle_key, start_loc, drawer_loc = None, None, None, None, None
        for k, v in positions.items():
            if 'drawer' in k and not drawer_key:
                drawer_key = k
                drawer_loc = v
            elif 'handle' in k and not handle_key:
                handle_key = k
            elif 'object' in k or 'ball' in k:
                if not object_key:
                    object_key = k
                    start_loc = v

        # Step 1: Move to object (if not already there)
        try:
            print(f"[Plan] Moving to {start_loc} ...")
            obs, reward, done = execute_go(env, task, start_loc, start_loc)
        except Exception as e:
            print(f"[Plan] execute_go failed: {e}")

        # Step 2: Pick object
        try:
            print(f"[Plan] Picking object {object_key}...")
            obs, reward, done = execute_pick(env, task, object_key, start_loc)
        except Exception as e:
            print(f"[Plan] execute_pick failed: {e}")

        # Step 3: Move to drawer location
        try:
            print(f"[Plan] Moving to drawer {drawer_loc} ...")
            obs, reward, done = execute_go(env, task, start_loc, drawer_loc)
        except Exception as e:
            print(f"[Plan] execute_go to drawer failed: {e}")

        # Step 4: Pull handle to open drawer (if handle exists)
        if handle_key:
            try:
                print(f"[Plan] Pulling handle {handle_key} to open drawer {drawer_key} ...")
                obs, reward, done = execute_pick(env, task, handle_key, drawer_loc)
                obs, reward, done = execute_pull(env, task, drawer_key, handle_key, drawer_loc)
            except Exception as e:
                print(f"[Plan] Open drawer failed: {e}")

        # Step 5: Place object in drawer
        try:
            print(f"[Plan] Placing object {object_key} in drawer {drawer_key} ...")
            obs, reward, done = execute_place(env, task, object_key, drawer_key, drawer_loc)
        except Exception as e:
            print(f"[Plan] execute_place failed: {e}")

        # Step 6: Push to close drawer
        try:
            print(f"[Plan] Closing drawer {drawer_key} ...")
            obs, reward, done = execute_push(env, task, drawer_key, drawer_loc)
        except Exception as e:
            print(f"[Plan] execute_push failed: {e}")

        print("[Plan] Oracle plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
