# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and check if the robot can be at 'floor' or similar locations.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to use execute_go to move the robot to 'floor' if such a location exists
        # This will help us determine if (robot-at floor) is a valid predicate in the environment

        # Attempt to find a location named 'floor' in the positions dictionary
        floor_location = None
        for key in positions.keys():
            if 'floor' in key.lower():
                floor_location = key
                break

        # If 'floor' is not found, just pick any location as a fallback for exploration
        if floor_location is None and len(positions) > 0:
            floor_location = list(positions.keys())[0]

        # Get current robot location (if available)
        robot_location = None
        for key in positions.keys():
            if 'robot' in key.lower():
                robot_location = key
                break

        # If robot location is not found, pick any location as a fallback
        if robot_location is None and len(positions) > 0:
            robot_location = list(positions.keys())[0]

        print(f"[Exploration] Attempting to move robot from '{robot_location}' to '{floor_location}' to test (robot-at floor) predicate.")

        # Use execute_go to move the robot and observe the result
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=floor_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] execute_go from '{robot_location}' to '{floor_location}' executed.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # After moving, check if the robot is at the expected location
        # This can help confirm if (robot-at floor) is a valid predicate
        try:
            current_positions = get_object_positions()
            print(f"[Exploration] Current positions after execute_go: {current_positions}")
        except Exception as e:
            print(f"[Exploration] Error retrieving positions after move: {e}")

        # Additional exploration: Try other skills if needed to further probe the environment
        # For example, try execute_pick or execute_sweep at the 'floor' location
        try:
            for obj_name, obj_pos in positions.items():
                if obj_name != floor_location and 'object' in obj_name.lower():
                    print(f"[Exploration] Attempting to pick '{obj_name}' at '{floor_location}'")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj_name,
                        location_name=floor_location,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] execute_pick for '{obj_name}' at '{floor_location}' executed.")
                    break
        except Exception as e:
            print(f"[Exploration] Error during execute_pick: {e}")

        # Exploration phase complete
        print("[Exploration] Exploration phase complete. Check logs and environment state for missing predicate clues.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
