# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe the environment's response.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers in the environment
        # For demonstration, we will attempt to:
        # 1. Move to each location (execute_go)
        # 2. Pick up each object (execute_pick)
        # 3. Pull each drawer (execute_pull)
        # 4. Place objects (execute_place)
        # 5. Push drawers (execute_push)
        # 6. Sweep (execute_sweep)
        # 7. Use gripper (execute_gripper)
        # 8. Try to turn on the light (execute_push_switch) if available

        # The actual object and location names should be extracted from the observation or positions
        # For this generic code, we will use placeholder names and handle exceptions gracefully

        # Example object and location lists (replace with actual names from your environment)
        object_names = []
        drawer_names = []
        location_names = []
        switch_names = []

        # Try to extract object, drawer, location, and switch names from positions or descriptions
        try:
            if 'objects' in positions:
                object_names = positions['objects']
            if 'drawers' in positions:
                drawer_names = positions['drawers']
            if 'locations' in positions:
                location_names = positions['locations']
            if 'switches' in positions:
                switch_names = positions['switches']
        except Exception as e:
            print("[Exploration] Could not extract names from positions:", e)

        # Fallback: Try to parse from descriptions if available
        if not object_names and 'objects' in descriptions:
            object_names = descriptions['objects']
        if not drawer_names and 'drawers' in descriptions:
            drawer_names = descriptions['drawers']
        if not location_names and 'locations' in descriptions:
            location_names = descriptions['locations']
        if not switch_names and 'switches' in descriptions:
            switch_names = descriptions['switches']

        # If still empty, try to guess from keys in positions
        if not object_names:
            object_names = [k for k in positions.keys() if 'object' in k]
        if not drawer_names:
            drawer_names = [k for k in positions.keys() if 'drawer' in k]
        if not location_names:
            location_names = [k for k in positions.keys() if 'room' in k or 'location' in k]
        if not switch_names:
            switch_names = [k for k in positions.keys() if 'switch' in k]

        # Print what we found
        print("[Exploration] Objects:", object_names)
        print("[Exploration] Drawers:", drawer_names)
        print("[Exploration] Locations:", location_names)
        print("[Exploration] Switches:", switch_names)

        # 1. Try to turn on the light if possible (execute_push_switch)
        for switch in switch_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to turn on light at switch {switch} in location {loc}")
                    obs, reward, done = execute_push_switch(env, task, switch, loc)
                    if done:
                        print("[Exploration] Light turned on successfully.")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_push_switch failed for {switch} at {loc}: {e}")

        # 2. Try to move to each location (execute_go)
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Trying to move from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Moved from {from_loc} to {to_loc}")
                except Exception as e:
                    print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        # 3. Try to pick up each object (execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick up object {obj} at location {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Picked up {obj} at {loc}")
                        # Try to place it in each drawer
                        for drawer in drawer_names:
                            try:
                                print(f"[Exploration] Trying to place {obj} in drawer {drawer} at location {loc}")
                                obs, reward, done2 = execute_place(env, task, obj, drawer, loc)
                                if done2:
                                    print(f"[Exploration] Placed {obj} in {drawer} at {loc}")
                            except Exception as e:
                                print(f"[Exploration] execute_place failed for {obj} in {drawer} at {loc}: {e}")
                        # Try to sweep
                        try:
                            print(f"[Exploration] Trying to sweep {obj} at {loc}")
                            obs, reward, done3 = execute_sweep(env, task, obj, loc)
                            if done3:
                                print(f"[Exploration] Swept {obj} at {loc}")
                        except Exception as e:
                            print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")
                        # Try to use gripper
                        try:
                            print(f"[Exploration] Trying to use gripper after picking {obj}")
                            obs, reward, done4 = execute_gripper(env, task)
                            if done4:
                                print(f"[Exploration] Used gripper after picking {obj}")
                        except Exception as e:
                            print(f"[Exploration] execute_gripper failed after picking {obj}: {e}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # 4. Try to pull each drawer (execute_pull)
        for drawer in drawer_names:
            for handle in object_names:  # Assume handles are objects
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to pull drawer {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Pulled drawer {drawer} with handle {handle} at {loc}")
                            # Try to push the drawer closed
                            try:
                                print(f"[Exploration] Trying to push drawer {drawer} at {loc}")
                                obs, reward, done2 = execute_push(env, task, drawer, loc)
                                if done2:
                                    print(f"[Exploration] Pushed drawer {drawer} at {loc}")
                            except Exception as e:
                                print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")

        # 5. Try to rotate (execute_rotate) if available
        if 'execute_rotate' in available_skills:
            for obj in object_names:
                try:
                    print(f"[Exploration] Trying to rotate object {obj}")
                    obs, reward, done = execute_rotate(env, task, obj)
                    if done:
                        print(f"[Exploration] Rotated {obj}")
                except Exception as e:
                    print(f"[Exploration] execute_rotate failed for {obj}: {e}")

        # 6. Try to use gripper (execute_gripper) in general
        try:
            print(f"[Exploration] Trying to use gripper (general)")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Used gripper (general)")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed (general): {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze the logs and environment feedback to determine which predicate is missing.
        # This information can be used to update the domain/problem PDDL or to inform further planning.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()