# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and confirm the robot's location predicates.

        # Try to use execute_go to move the robot and observe the effect
        # Since we don't have the actual object names, we will attempt to use the available positions
        # and skill names to perform exploration.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to find two different locations to perform execute_go
        location_names = []
        for name, pos in positions.items():
            if 'location' in name or 'floor' in name or 'room' in name:
                location_names.append(name)
        # Fallback: if no explicit location names, just use all keys
        if len(location_names) < 2:
            location_names = list(positions.keys())

        # Ensure at least two locations for exploration
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
        else:
            print("[Exploration] Not enough locations found for exploration.")
            from_loc = None
            to_loc = None

        # Try to execute 'execute_go' if possible
        if from_loc and to_loc:
            print(f"[Exploration] Attempting to move robot from '{from_loc}' to '{to_loc}' using execute_go.")
            try:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_loc,
                    to_location=to_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_go completed from '{from_loc}' to '{to_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # Try to execute 'execute_pick' on an object at the current location
        object_names = []
        for name in positions.keys():
            if 'object' in name or 'ball' in name or 'handle' in name:
                object_names.append(name)
        if len(object_names) > 0 and from_loc:
            obj = object_names[0]
            print(f"[Exploration] Attempting to pick '{obj}' at '{from_loc}' using execute_pick.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=obj,
                    location=from_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_pick completed for '{obj}' at '{from_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

        # Try to execute 'execute_place' if possible
        drawer_names = []
        for name in positions.keys():
            if 'drawer' in name:
                drawer_names.append(name)
        if len(drawer_names) > 0 and len(object_names) > 0 and from_loc:
            obj = object_names[0]
            drawer = drawer_names[0]
            print(f"[Exploration] Attempting to place '{obj}' in '{drawer}' at '{from_loc}' using execute_place.")
            try:
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=obj,
                    drawer_name=drawer,
                    location=from_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_place completed for '{obj}' in '{drawer}' at '{from_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # Try to execute 'execute_push' on a drawer if possible
        if len(drawer_names) > 0 and from_loc:
            drawer = drawer_names[0]
            print(f"[Exploration] Attempting to push '{drawer}' at '{from_loc}' using execute_push.")
            try:
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer,
                    location=from_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_push completed for '{drawer}' at '{from_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # Try to execute 'execute_pull' on a handle if possible
        handle_names = []
        for name in positions.keys():
            if 'handle' in name:
                handle_names.append(name)
        if len(drawer_names) > 0 and len(handle_names) > 0 and from_loc:
            drawer = drawer_names[0]
            handle = handle_names[0]
            print(f"[Exploration] Attempting to pull '{drawer}' with handle '{handle}' at '{from_loc}' using execute_pull.")
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawer,
                    handle_name=handle,
                    location=from_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_pull completed for '{drawer}' with handle '{handle}' at '{from_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

        # Try to execute 'execute_sweep' on an object if possible
        if len(object_names) > 0 and from_loc:
            obj = object_names[0]
            print(f"[Exploration] Attempting to sweep '{obj}' at '{from_loc}' using execute_sweep.")
            try:
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    object_name=obj,
                    location=from_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_sweep completed for '{obj}' at '{from_loc}'.")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

        # Try to execute 'execute_gripper' if possible
        print(f"[Exploration] Attempting to execute_gripper.")
        try:
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] execute_gripper completed.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase completed. Check logs and environment state for missing predicate information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()