# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills and observe the environment to infer which predicate is missing.
        # We use only the provided skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For demonstration, we will attempt to execute each skill in a safe, exploratory manner.
        # We will print the result and any exceptions to help identify missing predicates.

        # Get all objects, drawers, locations, and switches from positions (if available)
        # This is a generic approach; adapt as needed for your environment
        objects = []
        drawers = []
        locations = []
        switches = []
        handles = []
        for name, info in positions.items():
            if 'type' in info:
                if info['type'] == 'object':
                    objects.append(name)
                elif info['type'] == 'drawer':
                    drawers.append(name)
                elif info['type'] == 'location':
                    locations.append(name)
                elif info['type'] == 'switch':
                    switches.append(name)
                elif info['type'] == 'handle':
                    handles.append(name)
            else:
                # Fallback: guess by name
                if 'drawer' in name:
                    drawers.append(name)
                elif 'switch' in name:
                    switches.append(name)
                elif 'handle' in name:
                    handles.append(name)
                elif 'loc' in name or 'room' in name:
                    locations.append(name)
                else:
                    objects.append(name)

        # If no locations found, try to infer from positions
        if not locations:
            for name in positions:
                if 'room' in name or 'loc' in name:
                    locations.append(name)

        # If still empty, just use all keys as possible locations
        if not locations:
            locations = list(positions.keys())

        # Print what we found
        print("[Exploration] Objects:", objects)
        print("[Exploration] Drawers:", drawers)
        print("[Exploration] Locations:", locations)
        print("[Exploration] Switches:", switches)
        print("[Exploration] Handles:", handles)

        # Try each skill in a safe way, catch exceptions, and print feedback
        # This will help us identify which predicate is missing by observing which actions fail and why

        # 1. Try execute_go between two locations
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print("[Exploration] execute_go failed:", e)

        # 2. Try execute_pick on an object at a location
        if objects and locations:
            obj = objects[0]
            loc = locations[0]
            try:
                print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                obs, reward, done = execute_pick(env, task, obj, loc)
                print("[Exploration] execute_pick succeeded.")
            except Exception as e:
                print("[Exploration] execute_pick failed:", e)

        # 3. Try execute_place (requires object held, drawer open, etc.)
        if objects and drawers and locations:
            obj = objects[0]
            drawer = drawers[0]
            loc = locations[0]
            try:
                print(f"[Exploration] Trying execute_place of {obj} into {drawer} at {loc}")
                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                print("[Exploration] execute_place succeeded.")
            except Exception as e:
                print("[Exploration] execute_place failed:", e)

        # 4. Try execute_push (close drawer)
        if drawers and locations:
            drawer = drawers[0]
            loc = locations[0]
            try:
                print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                obs, reward, done = execute_push(env, task, drawer, loc)
                print("[Exploration] execute_push succeeded.")
            except Exception as e:
                print("[Exploration] execute_push failed:", e)

        # 5. Try execute_pull (open drawer with handle)
        if drawers and handles and locations:
            drawer = drawers[0]
            handle = handles[0]
            loc = locations[0]
            try:
                print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle} at {loc}")
                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                print("[Exploration] execute_pull succeeded.")
            except Exception as e:
                print("[Exploration] execute_pull failed:", e)

        # 6. Try execute_sweep (sweep object at location)
        if objects and locations:
            obj = objects[0]
            loc = locations[0]
            try:
                print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                obs, reward, done = execute_sweep(env, task, obj, loc)
                print("[Exploration] execute_sweep succeeded.")
            except Exception as e:
                print("[Exploration] execute_sweep failed:", e)

        # 7. Try execute_gripper (no parameters)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        # 8. Try execute_push_switch (turn on light if available)
        if switches and locations:
            switch = switches[0]
            loc = locations[0]
            try:
                print(f"[Exploration] Trying execute_push_switch on {switch} at {loc}")
                obs, reward, done = execute_push_switch(env, task, switch, loc)
                print("[Exploration] execute_push_switch succeeded.")
            except Exception as e:
                print("[Exploration] execute_push_switch failed:", e)

        print("[Exploration] Exploration phase complete. Review above output for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
