# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills and observe the environment to infer which predicate is missing.
        # We use only the provided skills: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we attempt to execute each skill and observe the result.
        # This will help us identify which skill (and thus which predicate) is not supported or is missing a precondition.

        # We'll print the result of each skill attempt for debugging.
        # We'll use try/except to handle any errors gracefully.

        # For demonstration, we assume the following keys exist in positions:
        # - 'robot': robot's current location
        # - 'objects': list of object names
        # - 'drawers': list of drawer names
        # - 'switches': list of switch names
        # - 'locations': list of all locations

        # If these keys do not exist, we handle the KeyError and print a warning.
        try:
            robot_loc = positions.get('robot', None)
            objects = positions.get('objects', [])
            drawers = positions.get('drawers', [])
            switches = positions.get('switches', [])
            locations = positions.get('locations', [])
        except Exception as e:
            print("[Exploration] Could not extract environment keys from positions:", e)
            robot_loc = None
            objects = []
            drawers = []
            switches = []
            locations = []

        # For each skill, try to execute with plausible arguments.
        # We do not know the exact arguments, so we use the first available object/drawer/location as a test.
        # This is for exploration only.

        # 1. Try execute_pick
        if len(objects) > 0 and robot_loc is not None:
            obj = objects[0]
            try:
                print("[Exploration] Trying execute_pick with object:", obj, "at", robot_loc)
                obs, reward, done = execute_pick(env, task, obj, robot_loc)
                print("[Exploration] execute_pick succeeded.")
            except Exception as e:
                print("[Exploration] execute_pick failed:", e)
        else:
            print("[Exploration] Skipping execute_pick: no objects or robot location.")

        # 2. Try execute_place
        if len(objects) > 0 and len(drawers) > 0 and robot_loc is not None:
            obj = objects[0]
            drawer = drawers[0]
            try:
                print("[Exploration] Trying execute_place with object:", obj, "into drawer:", drawer, "at", robot_loc)
                obs, reward, done = execute_place(env, task, obj, drawer, robot_loc)
                print("[Exploration] execute_place succeeded.")
            except Exception as e:
                print("[Exploration] execute_place failed:", e)
        else:
            print("[Exploration] Skipping execute_place: missing objects, drawers, or robot location.")

        # 3. Try execute_push (drawer close)
        if len(drawers) > 0 and robot_loc is not None:
            drawer = drawers[0]
            try:
                print("[Exploration] Trying execute_push on drawer:", drawer, "at", robot_loc)
                obs, reward, done = execute_push(env, task, drawer, robot_loc)
                print("[Exploration] execute_push succeeded.")
            except Exception as e:
                print("[Exploration] execute_push failed:", e)
        else:
            print("[Exploration] Skipping execute_push: missing drawers or robot location.")

        # 4. Try execute_pull (drawer open)
        if len(drawers) > 0 and len(objects) > 0 and robot_loc is not None:
            drawer = drawers[0]
            handle = objects[0]  # Assume first object is a handle for exploration
            try:
                print("[Exploration] Trying execute_pull on drawer:", drawer, "with handle:", handle, "at", robot_loc)
                obs, reward, done = execute_pull(env, task, drawer, handle, robot_loc)
                print("[Exploration] execute_pull succeeded.")
            except Exception as e:
                print("[Exploration] execute_pull failed:", e)
        else:
            print("[Exploration] Skipping execute_pull: missing drawers, objects, or robot location.")

        # 5. Try execute_sweep
        if len(objects) > 0 and robot_loc is not None:
            obj = objects[0]
            try:
                print("[Exploration] Trying execute_sweep on object:", obj, "at", robot_loc)
                obs, reward, done = execute_sweep(env, task, obj, robot_loc)
                print("[Exploration] execute_sweep succeeded.")
            except Exception as e:
                print("[Exploration] execute_sweep failed:", e)
        else:
            print("[Exploration] Skipping execute_sweep: missing objects or robot location.")

        # 6. Try execute_rotate (if available)
        try:
            print("[Exploration] Trying execute_rotate (no arguments)")
            obs, reward, done = execute_rotate(env, task)
            print("[Exploration] execute_rotate succeeded.")
        except Exception as e:
            print("[Exploration] execute_rotate failed:", e)

        # 7. Try execute_go (move between locations)
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                print("[Exploration] Trying execute_go from", from_loc, "to", to_loc)
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print("[Exploration] execute_go failed:", e)
        else:
            print("[Exploration] Skipping execute_go: not enough locations.")

        # 8. Try execute_gripper (no arguments)
        try:
            print("[Exploration] Trying execute_gripper (no arguments)")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        print("[Exploration] Exploration phase complete. Review above outputs to identify missing predicates or unsupported actions.")

        # === End of Exploration Phase ===

        # After exploration, you would analyze the printed outputs to determine which predicate is missing
        # (e.g., if an action always fails due to a missing precondition, that predicate is likely missing).

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
