# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We use the exploration domain knowledge to try to discover which predicate is missing.
        # We attempt to execute exploration skills to gather more information about objects.

        # For demonstration, we try to identify, check temperature, weight, durability, and lock status
        # of all objects at all locations, using only available skills.

        # Since only predefined skills can be used, and exploration skills are not in the available skill list,
        # we focus on using the available skills to interact with the environment and observe any issues.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper', 'execute_push_swtich -> push']

        # Let's try to execute each available skill in a safe, exploratory way.
        # We will attempt to pick, place, push, pull, sweep, rotate, go, gripper, and push switch
        # on all objects/locations as appropriate, catching exceptions to see if any predicate is missing.

        # Get lists of objects, drawers, locations, switches from positions (if available)
        objects = []
        drawers = []
        locations = []
        handles = []
        switches = []
        for name, pos in positions.items():
            if "drawer" in name:
                drawers.append(name)
            elif "handle" in name:
                handles.append(name)
            elif "switch" in name:
                switches.append(name)
            elif "loc" in name or "room" in name or "table" in name:
                locations.append(name)
            else:
                objects.append(name)

        # Fallback: if no locations found, use all unique locations from positions
        if not locations:
            locations = list(set([v for k, v in positions.items() if isinstance(v, str)]))

        # Try to execute each skill and catch any errors
        for loc in locations:
            for obj in objects:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")
                try:
                    print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        for d in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_push on {d} at {loc}")
                    obs, reward, done = execute_push(env, task, d, loc)
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {d} at {loc}: {e}")

        for d in drawers:
            for h in handles:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_pull on {d} with handle {h} at {loc}")
                        obs, reward, done = execute_pull(env, task, d, h, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {d}, {h} at {loc}: {e}")

        for obj in objects:
            for d in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_place {obj} in {d} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, d, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj} in {d} at {loc}: {e}")

        for from_loc in locations:
            for to_loc in locations:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    except Exception as e:
                        print(f"[Exploration] execute_go failed from {from_loc} to {to_loc}: {e}")

        for s in switches:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_push_switch on {s} at {loc}")
                    obs, reward, done = execute_push_switch(env, task, s, loc)
                except Exception as e:
                    print(f"[Exploration] execute_push_switch failed for {s} at {loc}: {e}")

        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # After exploration, print a message to indicate completion
        print("[Exploration] Exploration phase complete. Check above for any missing predicate errors.")

        # === (Optional) Main Task Plan Execution ===
        # At this point, you would normally execute the oracle plan step-by-step using the available skills.
        # For this exploration task, the focus is on discovering missing predicates via skill invocation and error feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
