# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore and confirm the robot's location using available skills.

        # For this, we will use the available skills to move the robot and check its state.
        # Since we do not have a direct exploration skill, we use the available skills to try to move or interact.

        # Let's assume 'floor' is a location in the environment.
        # Try to use 'execute_go' to move the robot to 'floor' if not already there.

        # Find the robot's current location and the list of locations
        robot_location = None
        locations = []
        for name, pos in positions.items():
            if 'robot' in name:
                robot_location = name
            if 'floor' in name or 'room' in name or 'location' in name:
                locations.append(name)
        if 'floor' not in locations:
            locations.append('floor')  # Ensure 'floor' is in the list

        # If we cannot determine the robot's location, default to 'floor'
        if robot_location is None:
            robot_location = 'floor'

        # Exploration: Try to move the robot to 'floor' using execute_go
        try:
            print("[Exploration] Attempting to move robot to 'floor' to check predicate (robot-at floor)")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location='floor',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Robot moved to 'floor'. Predicate (robot-at floor) is now true.")
        except Exception as e:
            print(f"[Exploration] Could not move robot to 'floor': {e}")

        # After exploration, proceed with the main plan (oracle plan)
        # Example: Pick up an object from the floor and place it in a drawer

        # Find an object on the floor and a drawer
        object_on_floor = None
        drawer = None
        for name in positions:
            if 'object' in name or 'ball' in name:
                object_on_floor = name
            if 'drawer' in name:
                drawer = name
        if object_on_floor is None:
            print("[Task] No object found on the floor for pick action.")
        if drawer is None:
            print("[Task] No drawer found for place action.")

        # 1. Turn on the light if the room is dark (execute_push_switch)
        try:
            print("[Task] Turning on the light (if needed)...")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name='switch',
                location='floor',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Light turned on.")
        except Exception as e:
            print(f"[Task] Could not turn on the light: {e}")

        # 2. Pick up the object from the floor (execute_pick)
        try:
            print(f"[Task] Picking up {object_on_floor} from floor...")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_on_floor,
                location='floor',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked up {object_on_floor}.")
        except Exception as e:
            print(f"[Task] Could not pick up {object_on_floor}: {e}")

        # 3. Move to the drawer location (execute_go)
        try:
            print(f"[Task] Moving to drawer location: {drawer}...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location='floor',
                to_location=drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Moved to {drawer}.")
        except Exception as e:
            print(f"[Task] Could not move to {drawer}: {e}")

        # 4. Open the drawer (execute_pull)
        try:
            print(f"[Task] Opening the drawer: {drawer}...")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer,
                handle_name='handle',
                location=drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Drawer {drawer} opened.")
        except Exception as e:
            print(f"[Task] Could not open drawer {drawer}: {e}")

        # 5. Place the object in the drawer (execute_place)
        try:
            print(f"[Task] Placing {object_on_floor} in {drawer}...")
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_on_floor,
                drawer_name=drawer,
                location=drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Placed {object_on_floor} in {drawer}.")
        except Exception as e:
            print(f"[Task] Could not place {object_on_floor} in {drawer}: {e}")

        # 6. Close the drawer (execute_push)
        try:
            print(f"[Task] Closing the drawer: {drawer}...")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer,
                location=drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Drawer {drawer} closed.")
        except Exception as e:
            print(f"[Task] Could not close drawer {drawer}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()