# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the presence/absence of this predicate.

        # For exploration, we will use available skills to move the robot and observe the state.
        # Since only predefined skills can be used, we will use 'execute_go' to move the robot.

        # Let's try to move the robot to all known locations and print the result.
        # This will help us determine if (robot-at floor) is a valid predicate/state.

        # Assume positions contains location names as keys, e.g., 'floor', 'drawer_area', etc.
        location_names = [name for name in positions if 'location' in name or 'floor' in name or 'room' in name]
        if not location_names:
            # Fallback: try some default location names
            location_names = ['floor']

        # Try to move the robot to each location using execute_go
        # We need to know the robot's current location; assume it's in obs or descriptions
        current_location = None
        for desc in descriptions:
            if 'robot-at' in desc:
                # e.g., desc: '(robot-at floor)'
                parts = desc.replace('(','').replace(')','').split()
                if len(parts) == 2:
                    current_location = parts[1]
                    break
        if current_location is None:
            # Fallback: try to get from obs or assume 'floor'
            current_location = 'floor'

        print(f"[Exploration] Robot initial location: {current_location}")
        for loc in location_names:
            if loc == current_location:
                continue
            try:
                print(f"[Exploration] Attempting to move robot from {current_location} to {loc} using execute_go...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {loc}.")
                current_location = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

        # After exploration, print out the robot's location to check if (robot-at floor) is valid
        # This helps us confirm the missing predicate

        # Optionally, print the current state
        try:
            current_obs = task.get_observation()
            print(f"[Exploration] Current observation after exploration: {current_obs}")
        except Exception as e:
            print(f"[Exploration] Could not retrieve observation: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate information.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, proceed with the main oracle plan using available skills.
        # For now, the focus is on exploration to find the missing predicate as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
