# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # For this, we will attempt to use the available skills in a systematic way to see which predicates are required.
        # Since we cannot define new skills, we will use the available ones and observe the environment after each action.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # For exploration, we will try to use each skill on each object/location as appropriate.
        # We will log the results and any exceptions to help identify missing predicates.

        # Get all objects and locations from positions
        object_names = [name for name in positions if 'object' in name or 'obj' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'drawer' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        switch_names = [name for name in positions if 'switch' in name]

        # Helper: Try a skill and catch errors
        def try_skill(skill_fn, *args, **kwargs):
            try:
                print(f"[Exploration] Trying {skill_fn.__name__} with args {args}")
                obs, reward, done = skill_fn(env, task, *args, **kwargs)
                print(f"[Exploration] {skill_fn.__name__} succeeded. Reward: {reward}, Done: {done}")
                return obs, reward, done
            except Exception as e:
                print(f"[Exploration] {skill_fn.__name__} failed with error: {e}")
                return None, None, False

        # 1. Try to turn on the light if possible (execute_push_switch)
        # The domain says this is only possible if (room-dark), (robot-free), (robot-at ?p)
        # Try for each switch and location
        for switch in switch_names:
            for loc in location_names:
                if hasattr(env, 'execute_push_switch'):
                    try_skill(env.execute_push_switch, switch, loc)
                elif 'execute_push_switch' in globals():
                    try_skill(execute_push_switch, switch, loc)
                else:
                    # The skill list says 'execute_push_swtich -> push', so try execute_push if available
                    if 'execute_push' in globals():
                        try_skill(execute_push, switch, loc)
                    else:
                        print("[Exploration] No execute_push_switch or execute_push skill found.")

        # 2. Try to pick up each object at each location
        for obj in object_names:
            for loc in location_names:
                if 'execute_pick' in globals():
                    try_skill(execute_pick, obj, loc)

        # 3. Try to open each drawer with each object as handle
        for drawer in drawer_names:
            for obj in object_names:
                for loc in location_names:
                    if 'execute_pull' in globals():
                        try_skill(execute_pull, drawer, obj, loc)

        # 4. Try to place each object in each drawer at each location
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    if 'execute_place' in globals():
                        try_skill(execute_place, obj, drawer, loc)

        # 5. Try to push (close) each drawer at each location
        for drawer in drawer_names:
            for loc in location_names:
                if 'execute_push' in globals():
                    try_skill(execute_push, drawer, loc)

        # 6. Try to move between locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc and 'execute_go' in globals():
                    try_skill(execute_go, from_loc, to_loc)

        # 7. Try to sweep each object at each location
        for obj in object_names:
            for loc in location_names:
                if 'execute_sweep' in globals():
                    try_skill(execute_sweep, obj, loc)

        # 8. Try to use gripper (if available)
        if 'execute_gripper' in globals():
            try_skill(execute_gripper)

        # 9. Try to rotate (if available)
        if 'execute_rotate' in globals():
            try_skill(execute_rotate)

        # After all exploration, print a message
        print("[Exploration] Completed all available skill trials. Check logs for failed actions and missing predicates.")

        # === End of Exploration Phase ===

        # At this point, based on the logs, you should be able to identify which predicate is missing
        # by observing which actions consistently fail and what preconditions are not met.

        # (Optional) You can now proceed to execute the actual oracle plan if available,
        # using the discovered predicates to inform your plan construction.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
