# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discovering Missing Predicate ===
        # Feedback indicates: (drawer-open drawer_bottom) is missing or not detected.
        # We need to explore the environment to determine the state of 'drawer_bottom' (open/closed).
        # The exploration domain provides actions to identify unknown predicates.
        # We will use available skills to interact with the drawer and infer its state.

        # Assumptions for object names (update as per your environment):
        # - The drawer of interest is 'drawer_bottom'
        # - The robot starts at some initial location, e.g., 'init_pos'
        # - The handle for the drawer is 'handle_bottom'
        # - The switch for the light is 'switch_main'
        # - The object to manipulate is 'target_object'
        # - The robot's initial location is 'robot_init'
        # - The location of the drawer is 'drawer_bottom_pos'
        # - The location of the switch is 'switch_pos'
        # - The location of the object is 'object_pos'

        # You may need to adjust these names based on your environment's actual object keys.
        try:
            drawer_name = 'drawer_bottom'
            handle_name = 'handle_bottom'
            switch_name = 'switch_main'
            robot_init = 'robot_init'
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            switch_pos = positions.get(switch_name, None)
            # For demonstration, pick the first object on the floor
            object_candidates = [k for k in positions.keys() if 'object' in k or 'ball' in k]
            if object_candidates:
                target_object = object_candidates[0]
                object_pos = positions[target_object]
            else:
                target_object = None
                object_pos = None
        except Exception as e:
            print("[Error] Failed to retrieve object positions:", e)
            shutdown_environment(env)
            return

        # Step 1: Ensure the room is bright (turn on the light if needed)
        # Try to execute the light switch skill if available
        try:
            # If the room is dark, turn on the light
            # execute_push_switch(switch_name, switch_pos)
            if 'execute_push_switch' in globals():
                obs, reward, done = execute_push_switch(env, task, switch_name, switch_pos)
                print("[Exploration] Turned on the light.")
        except Exception as e:
            print("[Warning] Could not execute 'execute_push_switch':", e)

        # Step 2: Move to the drawer location (if needed)
        try:
            if 'execute_go' in globals() and drawer_pos is not None:
                obs, reward, done = execute_go(env, task, robot_init, drawer_pos)
                print("[Exploration] Moved to drawer location.")
        except Exception as e:
            print("[Warning] Could not execute 'execute_go':", e)

        # Step 3: Try to pull the drawer to check if it is openable (explore lock/open state)
        # This will help us discover if the predicate (drawer-open drawer_bottom) is true or missing
        try:
            # First, pick the handle if required
            if 'execute_pick' in globals() and handle_pos is not None:
                obs, reward, done = execute_pick(env, task, handle_name, drawer_pos)
                print("[Exploration] Picked the drawer handle.")

            # Now, try to pull the drawer
            if 'execute_pull' in globals():
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_pos)
                print("[Exploration] Attempted to pull the drawer.")
        except Exception as e:
            print("[Warning] Could not execute 'execute_pull':", e)

        # Step 4: Check if the drawer is now open (predicate discovered)
        # This would be reflected in the environment state or observation
        try:
            # You may need to check the observation or environment state for 'drawer-open drawer_bottom'
            # For demonstration, print the current observation
            current_obs = task.get_observation()
            print("[Exploration] Current observation after pull:", current_obs)
            # If the predicate is still missing, further exploration may be needed
        except Exception as e:
            print("[Warning] Could not retrieve observation:", e)

        # Step 5: (Optional) Continue with the oracle plan if the predicate is now known
        # For example, pick an object and place it in the drawer if open
        try:
            if target_object is not None and object_pos is not None:
                # Move to object location
                if 'execute_go' in globals():
                    obs, reward, done = execute_go(env, task, drawer_pos, object_pos)
                    print("[Task] Moved to object location.")

                # Pick the object
                if 'execute_pick' in globals():
                    obs, reward, done = execute_pick(env, task, target_object, object_pos)
                    print("[Task] Picked the object.")

                # Move back to drawer
                if 'execute_go' in globals():
                    obs, reward, done = execute_go(env, task, object_pos, drawer_pos)
                    print("[Task] Returned to drawer.")

                # Place the object in the drawer
                if 'execute_place' in globals():
                    obs, reward, done = execute_place(env, task, target_object, drawer_name, drawer_pos)
                    print("[Task] Placed the object in the drawer.")
        except Exception as e:
            print("[Warning] Could not complete oracle plan steps:", e)

        # === End of Exploration and Plan Execution ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
