# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location, specifically (robot-at drawer-area).
        # We will use available skills to explore and confirm the robot's location predicates.

        # Step 1: Get all known locations from positions
        location_names = []
        for key, value in positions.items():
            if 'area' in key or 'room' in key or 'drawer' in key:
                location_names.append(key)
        # Fallback: If no area/room/drawer in keys, just use all keys
        if not location_names:
            location_names = list(positions.keys())

        # Step 2: Try to move the robot to each location using execute_go
        #         and check if the environment accepts the action.
        #         This will help us determine if (robot-at <location>) is a valid predicate.

        # For demonstration, we assume the robot starts at the first location.
        # We'll try to move to 'drawer-area' if it exists.
        robot_current_location = None
        if location_names:
            robot_current_location = location_names[0]
        drawer_area = None
        for loc in location_names:
            if 'drawer-area' in loc:
                drawer_area = loc
                break

        # If drawer-area is not found, just pick another location for exploration
        if not drawer_area and len(location_names) > 1:
            drawer_area = location_names[1]

        # Step 3: Exploration using execute_go
        # Only use predefined skills; do not define new ones.
        # Try to move the robot to drawer-area and observe if the predicate is accepted.

        print("[Exploration] Attempting to move robot to drawer-area to check predicate (robot-at drawer-area).")
        try:
            if robot_current_location and drawer_area and robot_current_location != drawer_area:
                # Use execute_go skill
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=drawer_area,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved from {robot_current_location} to {drawer_area}.")
                robot_current_location = drawer_area
            else:
                print("[Exploration] Could not determine distinct drawer-area for exploration.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # Step 4: Try to pick or interact at the new location to further confirm predicate usage
        # Use execute_pick or execute_sweep if available
        # We'll attempt to pick any object at the drawer-area

        # Find an object at the drawer-area
        object_to_pick = None
        for obj_name, pos in positions.items():
            if isinstance(pos, dict) and 'location' in pos:
                if pos['location'] == drawer_area:
                    object_to_pick = obj_name
                    break

        # Fallback: Just pick any object
        if not object_to_pick:
            for obj_name in positions.keys():
                if obj_name != drawer_area:
                    object_to_pick = obj_name
                    break

        if object_to_pick and robot_current_location:
            print(f"[Exploration] Attempting to pick {object_to_pick} at {robot_current_location}.")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_to_pick,
                    location=robot_current_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked {object_to_pick} at {robot_current_location}.")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick: {e}")
        else:
            print("[Exploration] No suitable object found for picking at drawer-area.")

        # === End of Exploration Phase ===

        # You can add further exploration or plan execution here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()