# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills and observe the environment to infer which predicate is missing.
        # We use only the provided skills: 
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # For exploration, we attempt to execute each skill in a safe, observable way.
        # We log the result and any exceptions to help identify missing predicates.

        # Get all objects, drawers, locations, switches from positions (if available)
        # This is a generic approach; adapt as needed for your environment
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name]
        switch_names = [name for name in positions if 'switch' in name]

        # Fallbacks if not found
        if not object_names:
            object_names = list(positions.keys())
        if not location_names:
            location_names = list(positions.keys())

        # Try to execute each skill and catch exceptions
        print("=== Exploration: Trying available skills to identify missing predicate ===")
        for skill in ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # Try picking the first object at its location
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_pick(env, task, obj, loc)
                        print(f"[Exploration] execute_pick({obj}, {loc}) succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_place':
                    # Try placing the first object into the first drawer at a location
                    if object_names and drawer_names and location_names:
                        obj = object_names[0]
                        drawer = drawer_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place({obj}, {drawer}, {loc}) succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_push':
                    # Try pushing the first drawer at a location
                    if drawer_names and location_names:
                        drawer = drawer_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        print(f"[Exploration] execute_push({drawer}, {loc}) succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_pull':
                    # Try pulling the first drawer handle at a location
                    if drawer_names and object_names and location_names:
                        drawer = drawer_names[0]
                        handle = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] execute_pull({drawer}, {handle}, {loc}) succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_sweep':
                    # Try sweeping the first object at a location
                    if object_names and location_names:
                        obj = object_names[0]
                        loc = location_names[0]
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        print(f"[Exploration] execute_sweep({obj}, {loc}) succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_rotate':
                    # If rotate takes no arguments
                    obs, reward, done = execute_rotate(env, task)
                    print(f"[Exploration] execute_rotate() succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_go':
                    # Try moving from one location to another
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"[Exploration] execute_go({from_loc}, {to_loc}) succeeded. Reward: {reward}, Done: {done}")
                elif skill == 'execute_gripper':
                    # If gripper takes no arguments
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper() succeeded. Reward: {reward}, Done: {done}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with exception: {e}")

        print("=== Exploration complete. Check above logs for clues about missing predicates or failed preconditions. ===")

        # After exploration, you can analyze which skills failed and which predicates might be missing.
        # This information can be used to update the domain/problem or to inform further debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()