# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (drawer-open drawer_bottom) is missing
        # We need to explore to determine the state of 'drawer_bottom' (e.g., is it open/closed/locked/unlocked?)
        # We'll use available skills to try to open the drawer and observe the result.

        # For this example, let's assume the following object names and locations:
        # - drawer_bottom: the drawer to be explored
        # - handle_bottom: the handle object for the drawer
        # - switch: the light switch object
        # - robot initial location: 'init_pos'
        # - drawer location: 'drawer_pos'
        # - switch location: 'switch_pos'

        # You may need to adjust these names based on your environment's object naming conventions.
        try:
            drawer_name = 'drawer_bottom'
            handle_name = 'handle_bottom'
            switch_name = 'switch'
            # Get locations from positions dict
            drawer_pos = positions.get(drawer_name, None)
            handle_pos = positions.get(handle_name, None)
            switch_pos = positions.get(switch_name, None)
            # Fallbacks if not found
            if drawer_pos is None:
                for k in positions:
                    if 'drawer' in k:
                        drawer_name = k
                        drawer_pos = positions[k]
                        break
            if handle_pos is None:
                for k in positions:
                    if 'handle' in k:
                        handle_name = k
                        handle_pos = positions[k]
                        break
            if switch_pos is None:
                for k in positions:
                    if 'switch' in k:
                        switch_name = k
                        switch_pos = positions[k]
                        break
            # Assume robot starts at some initial position
            robot_pos = None
            for k in positions:
                if 'robot' in k or 'init' in k:
                    robot_pos = positions[k]
                    break
            if robot_pos is None:
                robot_pos = (0, 0, 0)  # fallback

            # 1. If the room is dark, turn on the light
            # (Assume we can check this from obs or just always try)
            try:
                obs, reward, done = execute_push_switch(
                    env, task, switch_name, switch_pos
                )
                print("[Exploration] Turned on the light.")
            except Exception as e:
                print("[Exploration] Light may already be on or switch action failed:", e)

            # 2. Move to the handle location if not already there
            try:
                obs, reward, done = execute_go(
                    env, task, robot_pos, handle_pos
                )
                print("[Exploration] Moved to handle location.")
            except Exception as e:
                print("[Exploration] Move to handle failed (may already be there):", e)

            # 3. Try to pick the handle (if not already holding)
            try:
                obs, reward, done = execute_pick(
                    env, task, handle_name, handle_pos
                )
                print("[Exploration] Picked up the handle.")
            except Exception as e:
                print("[Exploration] Pick handle failed (may already be holding):", e)

            # 4. Try to pull the drawer open
            try:
                obs, reward, done = execute_pull(
                    env, task, drawer_name, handle_name, handle_pos
                )
                print("[Exploration] Pulled the drawer open.")
            except Exception as e:
                print("[Exploration] Pull drawer failed (may already be open):", e)

            # 5. At this point, we can check if the predicate (drawer-open drawer_bottom) is now true
            # (In a real system, you would check the state, but here we just log the attempt)
            print("[Exploration] Attempted to open drawer. If successful, (drawer-open drawer_bottom) should now hold.")

        except Exception as e:
            print("[Exploration] Error during exploration phase:", e)

        # === Main Task Plan (After Exploration) ===
        # At this point, you can proceed with the rest of your oracle plan, now that the missing predicate is resolved.
        # For example, you might want to place an object into the drawer, etc.
        # Here is a generic example of using the available skills:

        # Example: Place an object into the now-open drawer
        try:
            # 1. Pick up an object from the floor (find any object on the floor)
            object_to_pick = None
            object_pos = None
            for obj_name, pos in positions.items():
                if 'object' in obj_name or 'item' in obj_name:
                    object_to_pick = obj_name
                    object_pos = pos
                    break
            if object_to_pick is not None and object_pos is not None:
                obs, reward, done = execute_go(
                    env, task, handle_pos, object_pos
                )
                print("[Task] Moved to object location.")
                obs, reward, done = execute_pick(
                    env, task, object_to_pick, object_pos
                )
                print(f"[Task] Picked up {object_to_pick}.")
                # Move back to drawer
                obs, reward, done = execute_go(
                    env, task, object_pos, drawer_pos
                )
                print("[Task] Moved to drawer location.")
                # Place object in drawer
                obs, reward, done = execute_place(
                    env, task, object_to_pick, drawer_name, drawer_pos
                )
                print(f"[Task] Placed {object_to_pick} in {drawer_name}.")
            else:
                print("[Task] No suitable object found to pick and place.")

        except Exception as e:
            print("[Task] Error during main task plan:", e)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
