# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (room-bright) is missing; we need to ensure the room is bright before proceeding.
        # According to the domain, 'execute_push_switch' is the action to turn on the light (make room-bright true).
        # We will attempt to execute this action if the room is not already bright.

        # For demonstration, we assume the switch and its location are known from positions.
        # You may need to adjust these keys based on your environment's object naming.
        switch_name = None
        switch_location = None
        for obj_name, pos in positions.items():
            if 'switch' in obj_name.lower():
                switch_name = obj_name
                switch_location = pos
                break

        # If switch is not found, fallback to a default name/location (may need adjustment)
        if switch_name is None:
            switch_name = 'switch'
            switch_location = positions.get('switch', None)

        # Find robot's current location
        robot_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name.lower():
                robot_location = pos
                break

        # If robot location is not found, fallback to a default
        if robot_location is None:
            robot_location = positions.get('robot', None)

        # 1. Move robot to switch location if not already there
        try:
            if robot_location is not None and switch_location is not None and robot_location != switch_location:
                # Use execute_go to move robot (assuming location names are available)
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=switch_location
                )
                print(f"[Exploration] Robot moved to switch location: {switch_location}")
        except Exception as e:
            print(f"[Exploration] Error moving to switch: {e}")

        # 2. Execute push switch to turn on the light (room-bright)
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                switch_location
            )
            print(f"[Exploration] Executed push switch: {switch_name} at {switch_location}")
        except Exception as e:
            print(f"[Exploration] Error executing push switch: {e}")

        # === Main Task Plan ===
        # After ensuring (room-bright), proceed with the oracle plan using only available skills.
        # The following is a generic example; you must adapt object names and locations to your environment.

        # Example: Pick up an object from the floor and place it in a drawer
        # 1. Find an object on the floor and its location
        object_to_pick = None
        object_location = None
        for obj_name, pos in positions.items():
            if 'object' in obj_name.lower() or 'ball' in obj_name.lower():
                object_to_pick = obj_name
                object_location = pos
                break

        # 2. Move robot to object location if not already there
        try:
            if object_location is not None and switch_location is not None and object_location != switch_location:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=switch_location,
                    to_location=object_location
                )
                print(f"[Task] Robot moved to object location: {object_location}")
        except Exception as e:
            print(f"[Task] Error moving to object: {e}")

        # 3. Pick up the object
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_to_pick,
                object_location
            )
            print(f"[Task] Picked up object: {object_to_pick} at {object_location}")
        except Exception as e:
            print(f"[Task] Error picking up object: {e}")

        # 4. Find a drawer and its location
        drawer_name = None
        drawer_location = None
        for obj_name, pos in positions.items():
            if 'drawer' in obj_name.lower():
                drawer_name = obj_name
                drawer_location = pos
                break

        # 5. Move robot to drawer location if not already there
        try:
            if drawer_location is not None and object_location is not None and drawer_location != object_location:
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=object_location,
                    to_location=drawer_location
                )
                print(f"[Task] Robot moved to drawer location: {drawer_location}")
        except Exception as e:
            print(f"[Task] Error moving to drawer: {e}")

        # 6. Open the drawer (requires handle object, which must be found)
        handle_name = None
        for obj_name in positions.keys():
            if 'handle' in obj_name.lower():
                handle_name = obj_name
                break

        # Pick up the handle if needed
        if handle_name is not None:
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    handle_name,
                    drawer_location
                )
                print(f"[Task] Picked up handle: {handle_name}")
            except Exception as e:
                print(f"[Task] Error picking up handle: {e}")

            # Pull the drawer open
            try:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name,
                    handle_name,
                    drawer_location
                )
                print(f"[Task] Pulled open drawer: {drawer_name}")
            except Exception as e:
                print(f"[Task] Error pulling drawer: {e}")

        # 7. Place the object in the drawer
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_to_pick,
                drawer_name,
                drawer_location
            )
            print(f"[Task] Placed object: {object_to_pick} in drawer: {drawer_name}")
        except Exception as e:
            print(f"[Task] Error placing object in drawer: {e}")

        # 8. Close the drawer
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name,
                drawer_location
            )
            print(f"[Task] Closed drawer: {drawer_name}")
        except Exception as e:
            print(f"[Task] Error closing drawer: {e}")

        print("[Task] Task completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
