# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try each available skill to see which predicate is missing.
        # This is done by attempting to execute each skill and catching any errors or failures.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to execute each skill with plausible arguments.
        # In a real scenario, you would use the actual objects/locations from the environment.
        # Here, we use the positions dictionary and try to infer object/location names.

        # Extract object and location names from positions
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        switch_names = []

        for name in positions.keys():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'switch' in name:
                switch_names.append(name)
            elif 'loc' in name or 'room' in name or 'area' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallbacks if not found
        if not location_names:
            location_names = ['location1', 'location2']
        if not object_names:
            object_names = ['object1', 'object2']
        if not drawer_names:
            drawer_names = ['drawer1']
        if not handle_names:
            handle_names = ['handle1']
        if not switch_names:
            switch_names = ['switch1']

        # Use the first available for each type
        obj = object_names[0]
        obj2 = object_names[1] if len(object_names) > 1 else object_names[0]
        loc = location_names[0]
        loc2 = location_names[1] if len(location_names) > 1 else location_names[0]
        drawer = drawer_names[0]
        handle = handle_names[0]
        switch = switch_names[0]

        # Track which skills fail due to missing predicates
        failed_skills = []

        print("=== Exploration Phase: Testing available skills for missing predicates ===")
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # execute_pick(obj, location)
                    obs, reward, done = execute_pick(env, task, obj, loc)
                elif skill == 'execute_place':
                    # execute_place(obj, drawer, location)
                    obs, reward, done = execute_place(env, task, obj, drawer, loc)
                elif skill == 'execute_push':
                    # execute_push(drawer, location)
                    obs, reward, done = execute_push(env, task, drawer, loc)
                elif skill == 'execute_pull':
                    # execute_pull(drawer, handle, location)
                    obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                elif skill == 'execute_sweep':
                    # execute_sweep(obj, location)
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                elif skill == 'execute_rotate':
                    # execute_rotate(obj, location)
                    obs, reward, done = execute_rotate(env, task, obj, loc)
                elif skill == 'execute_go':
                    # execute_go(from_location, to_location)
                    obs, reward, done = execute_go(env, task, loc, loc2)
                elif skill == 'execute_gripper':
                    # execute_gripper()
                    obs, reward, done = execute_gripper(env, task)
                else:
                    print(f"[Exploration] Unknown skill: {skill}")
                    continue
                print(f"[Exploration] Skill {skill} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")
                failed_skills.append((skill, str(e)))

        print("=== Exploration Results ===")
        if failed_skills:
            print("Some skills failed, possibly due to missing predicates or preconditions:")
            for skill, error in failed_skills:
                print(f"  - {skill}: {error}")
        else:
            print("All skills executed successfully. No missing predicates detected in this phase.")

        # === End of Exploration Phase ===

        # (Optional) Here you would proceed to the main oracle plan execution
        # after identifying and fixing missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()