# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at handle_bottom_pose)
        # The robot needs to explore/visit 'handle_bottom_pose' to discover missing predicates (e.g., identify handle, check lock, etc.)
        # We use available skills to move the robot to 'handle_bottom_pose' and trigger any exploration/identification.

        # Try to get the robot's current location and the handle's location
        try:
            robot_pos = positions.get('robot', None)
            handle_bottom_pose = positions.get('handle_bottom_pose', None)
        except Exception as e:
            print("[Error] Could not retrieve positions:", e)
            robot_pos = None
            handle_bottom_pose = None

        # If positions are not available, skip exploration
        if robot_pos is not None and handle_bottom_pose is not None:
            print("[Exploration] Moving robot to handle_bottom_pose for exploration.")
            try:
                # Use the available skill 'execute_go' to move the robot
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_pos,
                    to_location=handle_bottom_pose
                )
                print("[Exploration] Robot moved to handle_bottom_pose.")
            except Exception as e:
                print("[Exploration Error] Failed to move robot to handle_bottom_pose:", e)
        else:
            print("[Exploration] handle_bottom_pose or robot position not found in positions. Skipping exploration.")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the main plan using only available skills.
        # Example: Pick up an object, open drawer, place object, etc.
        # The actual plan should be filled in according to the oracle plan and environment specifics.

        # Example placeholders for main plan steps (replace with actual plan as needed):

        # 1. Turn on the light if the room is dark
        try:
            switch_pos = positions.get('switch', None)
            robot_pos = positions.get('robot', None)
            if switch_pos is not None and robot_pos is not None:
                print("[Task] Turning on the light (if needed).")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch=switch_pos,
                    location=robot_pos
                )
        except Exception as e:
            print("[Task] Skipping light switch action:", e)

        # 2. Pick up the handle if required
        try:
            handle_obj = positions.get('handle', None)
            handle_location = positions.get('handle_location', None)
            if handle_obj is not None and handle_location is not None:
                print("[Task] Picking up the handle.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=handle_obj,
                    location=handle_location
                )
        except Exception as e:
            print("[Task] Skipping pick handle action:", e)

        # 3. Pull the drawer using the handle
        try:
            drawer = positions.get('drawer', None)
            handle_obj = positions.get('handle', None)
            drawer_location = positions.get('drawer_location', None)
            if drawer is not None and handle_obj is not None and drawer_location is not None:
                print("[Task] Pulling the drawer open.")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer,
                    handle=handle_obj,
                    location=drawer_location
                )
        except Exception as e:
            print("[Task] Skipping pull drawer action:", e)

        # 4. Place an object into the drawer
        try:
            obj_to_place = positions.get('object_to_place', None)
            drawer = positions.get('drawer', None)
            drawer_location = positions.get('drawer_location', None)
            if obj_to_place is not None and drawer is not None and drawer_location is not None:
                print("[Task] Placing object into the drawer.")
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj=obj_to_place,
                    drawer=drawer,
                    location=drawer_location
                )
        except Exception as e:
            print("[Task] Skipping place object action:", e)

        # 5. Push the drawer closed
        try:
            drawer = positions.get('drawer', None)
            drawer_location = positions.get('drawer_location', None)
            if drawer is not None and drawer_location is not None:
                print("[Task] Pushing the drawer closed.")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer,
                    location=drawer_location
                )
        except Exception as e:
            print("[Task] Skipping push drawer action:", e)

        # Add more plan steps as needed, using only the available skills and positions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
