# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try each available skill and observe the environment for changes.
        # This will help us infer which predicate is missing or required.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we will attempt to execute each skill with plausible arguments.
        # We will log the outcome and any exceptions to help identify missing predicates.

        # Get all objects, drawers, locations, switches from positions (if available)
        # For this example, we assume positions dict contains keys for 'objects', 'drawers', 'locations', 'switches'
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        switches = positions.get('switches', [])

        # Helper: Try a skill and catch errors
        def try_skill(skill_fn, *args, **kwargs):
            try:
                print(f"[Exploration] Trying {skill_fn.__name__} with args {args} kwargs {kwargs}")
                obs, reward, done = skill_fn(env, task, *args, **kwargs)
                print(f"[Exploration] {skill_fn.__name__} succeeded. Reward: {reward}, Done: {done}")
                return True
            except Exception as e:
                print(f"[Exploration] {skill_fn.__name__} failed with error: {e}")
                return False

        # 1. Try turning on the light (execute_push_switch) if available
        if 'execute_push_switch' in globals():
            for s in switches:
                for p in locations:
                    try_skill(execute_push_switch, s, p)

        # 2. Try moving (execute_go) between locations
        if 'execute_go' in globals() and len(locations) >= 2:
            for i in range(len(locations)):
                for j in range(len(locations)):
                    if i != j:
                        try_skill(execute_go, locations[i], locations[j])

        # 3. Try picking up each object (execute_pick)
        if 'execute_pick' in globals():
            for o in objects:
                for p in locations:
                    try_skill(execute_pick, o, p)

        # 4. Try placing each object into each drawer (execute_place)
        if 'execute_place' in globals():
            for o in objects:
                for d in drawers:
                    for p in locations:
                        try_skill(execute_place, o, d, p)

        # 5. Try pushing and pulling drawers (execute_push, execute_pull)
        if 'execute_push' in globals():
            for d in drawers:
                for p in locations:
                    try_skill(execute_push, d, p)
        if 'execute_pull' in globals():
            for d in drawers:
                for h in objects:  # handles are objects
                    for p in locations:
                        try_skill(execute_pull, d, h, p)

        # 6. Try sweeping and gripper actions
        if 'execute_sweep' in globals():
            for o in objects:
                for p in locations:
                    try_skill(execute_sweep, o, p)
        if 'execute_gripper' in globals():
            try_skill(execute_gripper)

        # 7. Try rotating if available
        if 'execute_rotate' in globals():
            # No parameters in domain, so just call
            try_skill(execute_rotate)

        print("[Exploration] Completed trying all available skills.")

        # === End of Exploration Phase ===
        # After running this, check the logs to see which actions failed and what errors were raised.
        # This will help you identify which predicate is missing in the domain/problem.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()