# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the robot's location and update our knowledge.

        # Try to use available skills to explore and identify the robot's location.
        # Since the available skills do not include a direct "explore" or "identify" action,
        # we will use the movement and observation skills to infer the missing predicate.

        # Step 1: Get all locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'location' in name:
                location_names.append(name)
        if not location_names:
            # Fallback: try to use any key that could be a location
            location_names = list(positions.keys())

        # Step 2: Try to move the robot to each location and check if the robot is at that location
        # Use execute_go if available
        # The skill signature is likely: execute_go(env, task, from_location, to_location, ...)
        # We'll try to move the robot to each location and print the result

        # For demonstration, we assume the robot starts at the first location
        if len(location_names) < 2:
            print("[Exploration] Not enough locations found for exploration.")
        else:
            from_location = location_names[0]
            for to_location in location_names[1:]:
                try:
                    print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} using execute_go.")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location,
                        to_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Robot moved to {to_location}.")
                    # After moving, check if the robot is at the new location
                    # This would be reflected in the observation or environment state
                    # For now, we print the feedback
                    # If the feedback is (robot-at floor), we have confirmed the missing predicate
                    # You may want to check obs or task.get_observation() for more details
                    current_obs = task.get_observation()
                    print(f"[Exploration] Current observation after moving to {to_location}: {current_obs}")
                    # Set from_location for the next move
                    from_location = to_location
                except Exception as e:
                    print(f"[Exploration] Failed to move robot from {from_location} to {to_location}: {e}")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate (e.g., (robot-at floor)), 
        # you can proceed to execute the main oracle plan using the available skills.
        # For now, the code focuses on the exploration phase as required by the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
