# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (drawer-open drawer_bottom) is missing.
        # We need to explore the state of 'drawer_bottom' to determine its predicates.

        # For this, we will attempt to use available skills to interact with 'drawer_bottom'
        # and observe its state, using only predefined skills.

        # Assume object and location names based on typical conventions
        drawer_name = 'drawer_bottom'
        # Try to get the handle object for the drawer, if available
        handle_name = None
        for obj in positions:
            if 'handle' in obj and drawer_name in obj:
                handle_name = obj
                break
        # Fallback: try a generic handle name
        if handle_name is None:
            handle_name = 'handle_' + drawer_name

        # Try to get the robot's current location
        robot_location = None
        for key in positions:
            if 'robot' in key:
                robot_location = key
                break
        # Fallback: use a default location
        if robot_location is None:
            robot_location = 'robot_base'

        # Try to get the location of the drawer
        drawer_location = None
        for key in positions:
            if drawer_name in key and 'location' in key:
                drawer_location = key
                break
        if drawer_location is None:
            drawer_location = drawer_name + '_location'

        # Try to get a switch for the light
        switch_name = None
        for obj in positions:
            if 'switch' in obj:
                switch_name = obj
                break
        if switch_name is None:
            switch_name = 'light_switch'
        # Assume the switch is at the same location as the robot or nearby
        switch_location = robot_location

        # === Step 1: If the room is dark, turn on the light ===
        # We need to ensure room-bright for most actions
        try:
            print("[Exploration] Trying to turn on the light if needed...")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                switch_location
            )
        except Exception as e:
            print("[Exploration] execute_push_switch failed or not needed:", e)

        # === Step 2: Try to open the drawer to check its state ===
        # To open the drawer, we need to:
        # 1. Pick the handle (if not already holding)
        # 2. Pull the handle (execute_pull)
        # 3. Observe if the drawer is open

        # Try picking the handle
        try:
            print(f"[Exploration] Trying to pick the handle: {handle_name} at {drawer_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                handle_name,
                drawer_location
            )
        except Exception as e:
            print("[Exploration] execute_pick on handle failed:", e)

        # Try pulling the handle to open the drawer
        try:
            print(f"[Exploration] Trying to pull the handle to open the drawer: {drawer_name}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                drawer_location
            )
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)

        # At this point, the predicate (drawer-open drawer_bottom) should be true if the drawer is open.
        # If not, the missing predicate is likely related to the drawer's state.

        # === Step 3: Exploration for Missing Predicate ===
        # If the drawer cannot be opened, or the state is ambiguous, try to use other skills to explore.
        # For example, try to push the drawer (close it), then open again.

        try:
            print(f"[Exploration] Trying to push (close) the drawer: {drawer_name}")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name,
                drawer_location
            )
        except Exception as e:
            print("[Exploration] execute_push failed:", e)

        # Try to open again after closing
        try:
            print(f"[Exploration] Trying to pick the handle again: {handle_name}")
            obs, reward, done = execute_pick(
                env,
                task,
                handle_name,
                drawer_location
            )
        except Exception as e:
            print("[Exploration] execute_pick on handle (second attempt) failed:", e)

        try:
            print(f"[Exploration] Trying to pull the handle again to open the drawer: {drawer_name}")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                drawer_location
            )
        except Exception as e:
            print("[Exploration] execute_pull (second attempt) failed:", e)

        # === Step 4: Exploration Complete ===
        print("[Exploration] Exploration phase complete. If the drawer is now open, the missing predicate is likely (drawer-open drawer_bottom).")

        # === Main Task Plan (Oracle Plan Execution) ===
        # At this point, you would continue with the oracle plan, using only the available skills.
        # For example, pick an object from the floor and place it in the drawer, etc.
        # The following is a placeholder for the main plan logic.

        # Example: Pick an object and place it in the drawer
        # Find an object on the floor
        object_on_floor = None
        for obj in positions:
            if 'object' in obj or 'item' in obj:
                object_on_floor = obj
                break
        if object_on_floor is None:
            object_on_floor = 'object_1'
        object_location = drawer_location  # Assume same location for simplicity

        # Pick the object
        try:
            print(f"[Task] Picking object: {object_on_floor} at {object_location}")
            obs, reward, done = execute_pick(
                env,
                task,
                object_on_floor,
                object_location
            )
        except Exception as e:
            print("[Task] execute_pick on object failed:", e)

        # Place the object in the drawer
        try:
            print(f"[Task] Placing object: {object_on_floor} in drawer: {drawer_name} at {drawer_location}")
            obs, reward, done = execute_place(
                env,
                task,
                object_on_floor,
                drawer_name,
                drawer_location
            )
        except Exception as e:
            print("[Task] execute_place failed:", e)

        print("[Task] Main plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()