# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The plan failed due to a missing predicate. We now perform exploration actions
        # to discover which predicate is missing, using only the available skills.

        # For demonstration, we attempt to use available skills in a systematic way
        # to probe the environment and log the results. This helps us infer which
        # predicate is missing (e.g., lock-known, identified, etc.).

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # For exploration, we try to execute each skill on each object/location
        # and observe the environment's response.

        # Get all objects and locations from positions
        object_names = [name for name in positions if 'object' in name or 'obj' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'drawer' in name]

        # Try to execute each skill and catch exceptions to infer missing predicates
        exploration_results = {}
        for skill in available_skills:
            for obj in object_names:
                for loc in location_names:
                    try:
                        # Map skill names to function calls
                        if skill == 'execute_pick':
                            print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                            obs, reward, done = execute_pick(
                                env, task, obj, loc
                            )
                            exploration_results[(skill, obj, loc)] = (obs, reward, done)
                        elif skill == 'execute_place':
                            # For place, need a drawer and an object
                            for drawer in location_names:
                                if 'drawer' in drawer:
                                    print(f"[Exploration] Trying execute_place for {obj} into {drawer} at {loc}")
                                    obs, reward, done = execute_place(
                                        env, task, obj, drawer, loc
                                    )
                                    exploration_results[(skill, obj, drawer, loc)] = (obs, reward, done)
                        elif skill == 'execute_push':
                            if 'drawer' in loc:
                                print(f"[Exploration] Trying execute_push on {loc}")
                                obs, reward, done = execute_push(
                                    env, task, loc
                                )
                                exploration_results[(skill, loc)] = (obs, reward, done)
                        elif skill == 'execute_pull':
                            # For pull, need a handle object and a drawer
                            for handle in object_names:
                                if 'handle' in handle:
                                    print(f"[Exploration] Trying execute_pull on {loc} with handle {handle}")
                                    obs, reward, done = execute_pull(
                                        env, task, loc, handle, loc
                                    )
                                    exploration_results[(skill, loc, handle)] = (obs, reward, done)
                        elif skill == 'execute_sweep':
                            print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                            obs, reward, done = execute_sweep(
                                env, task, obj, loc
                            )
                            exploration_results[(skill, obj, loc)] = (obs, reward, done)
                        elif skill == 'execute_rotate':
                            # If rotate is available, try on object
                            print(f"[Exploration] Trying execute_rotate on {obj}")
                            obs, reward, done = execute_rotate(
                                env, task, obj
                            )
                            exploration_results[(skill, obj)] = (obs, reward, done)
                        elif skill == 'execute_go':
                            for to_loc in location_names:
                                if to_loc != loc:
                                    print(f"[Exploration] Trying execute_go from {loc} to {to_loc}")
                                    obs, reward, done = execute_go(
                                        env, task, loc, to_loc
                                    )
                                    exploration_results[(skill, loc, to_loc)] = (obs, reward, done)
                        elif skill == 'execute_gripper':
                            print(f"[Exploration] Trying execute_gripper")
                            obs, reward, done = execute_gripper(
                                env, task
                            )
                            exploration_results[(skill,)] = (obs, reward, done)
                        elif skill.startswith('execute_push_swtich'):
                            # This is likely a typo, but if available, try to push a switch
                            for switch in object_names:
                                if 'switch' in switch:
                                    print(f"[Exploration] Trying execute_push_switch on {switch} at {loc}")
                                    obs, reward, done = execute_push_switch(
                                        env, task, switch, loc
                                    )
                                    exploration_results[(skill, switch, loc)] = (obs, reward, done)
                    except Exception as e:
                        print(f"[Exploration] Skill {skill} failed on ({obj}, {loc}): {e}")
                        exploration_results[(skill, obj, loc)] = str(e)
                        continue

        # After exploration, print summary of results
        print("===== Exploration Results Summary =====")
        for k, v in exploration_results.items():
            print(f"Skill/Args: {k} -> Result: {v}")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate from exploration,
        # you would update your plan or inform the planner.
        # For now, the code demonstrates how to systematically explore
        # the environment using only the available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()