# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The plan failed due to a missing predicate. We need to explore the environment
        # using only the available skills to discover which predicate is missing.
        # We will attempt to execute each available skill and observe the environment's response.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will try to execute each skill with plausible arguments.
        # In a real scenario, you would parse the domain/problem to get actual objects/locations.
        # Here, we use the positions dictionary and try to infer object/location names.

        # Extract object and location names from positions
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        switch_names = []

        for name in positions:
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'switch' in name:
                switch_names.append(name)
            elif 'loc' in name or 'room' in name or 'table' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallbacks if not found
        if not location_names:
            location_names = ['location1', 'location2']
        if not object_names:
            object_names = ['object1', 'object2']
        if not drawer_names:
            drawer_names = ['drawer1']
        if not handle_names:
            handle_names = ['handle1']
        if not switch_names:
            switch_names = ['switch1']

        # Assume robot starts at the first location
        current_location = location_names[0]

        # Try to turn on the light if possible (execute_push_switch is not in available_skills, so skip)
        # Try to move to another location if possible
        if len(location_names) > 1:
            try:
                print("[Exploration] Trying execute_go from", current_location, "to", location_names[1])
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_location,
                    to_location=location_names[1]
                )
                current_location = location_names[1]
            except Exception as e:
                print("[Exploration] execute_go failed:", e)

        # Try to pick up an object
        try:
            print("[Exploration] Trying execute_pick on", object_names[0], "at", current_location)
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_names[0],
                location_name=current_location
            )
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)

        # Try to open a drawer (pull)
        try:
            print("[Exploration] Trying execute_pull on", drawer_names[0], "with handle", handle_names[0], "at", current_location)
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_names[0],
                handle_name=handle_names[0],
                location_name=current_location
            )
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)

        # Try to place an object in a drawer
        try:
            print("[Exploration] Trying execute_place", object_names[0], "in", drawer_names[0], "at", current_location)
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_names[0],
                drawer_name=drawer_names[0],
                location_name=current_location
            )
        except Exception as e:
            print("[Exploration] execute_place failed:", e)

        # Try to push (close) a drawer
        try:
            print("[Exploration] Trying execute_push on", drawer_names[0], "at", current_location)
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_names[0],
                location_name=current_location
            )
        except Exception as e:
            print("[Exploration] execute_push failed:", e)

        # Try to sweep an object
        try:
            print("[Exploration] Trying execute_sweep on", object_names[0], "at", current_location)
            obs, reward, done = execute_sweep(
                env,
                task,
                object_name=object_names[0],
                location_name=current_location
            )
        except Exception as e:
            print("[Exploration] execute_sweep failed:", e)

        # Try to rotate (if implemented)
        try:
            print("[Exploration] Trying execute_rotate")
            obs, reward, done = execute_rotate(
                env,
                task
            )
        except Exception as e:
            print("[Exploration] execute_rotate failed:", e)

        # Try to use gripper (if implemented)
        try:
            print("[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # After exploration, you would analyze the feedback/logs to determine which predicate is missing.
        # This can be automated by parsing error messages or by observing which actions fail due to missing preconditions.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()