# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # Since we cannot define new skills, we use the available ones to interact with the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill in a safe, exploratory way
        # We will attempt to use each skill on each object/location, catching exceptions if preconditions are not met

        # Get all objects and locations from positions
        object_names = []
        location_names = []
        drawer_names = []
        switch_names = []
        handle_names = []
        for name, info in positions.items():
            if info.get('type') == 'object':
                object_names.append(name)
            elif info.get('type') == 'location':
                location_names.append(name)
            elif info.get('type') == 'drawer':
                drawer_names.append(name)
            elif info.get('type') == 'switch':
                switch_names.append(name)
            elif info.get('type') == 'handle':
                handle_names.append(name)

        # Fallback: If type info is not available, guess by name
        if not object_names:
            for name in positions.keys():
                if 'object' in name or 'ball' in name or 'item' in name:
                    object_names.append(name)
                elif 'drawer' in name:
                    drawer_names.append(name)
                elif 'switch' in name:
                    switch_names.append(name)
                elif 'handle' in name:
                    handle_names.append(name)
                elif 'room' in name or 'loc' in name:
                    location_names.append(name)

        # If still empty, just use all keys as objects
        if not object_names:
            object_names = list(positions.keys())

        # Try to move to each location (execute_go)
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Moved from {from_loc} to {to_loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")

        # Try to turn on the light (execute_push_switch)
        for switch in switch_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push_switch on {switch} at {loc}")
                    obs, reward, done = execute_push_switch(env, task, switch, loc)
                    if done:
                        print(f"[Exploration] Pushed switch {switch} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_push_switch failed: {e}")

        # Try to pick up each object (execute_pick)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Picked up {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # Try to place each object in each drawer (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_place {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Placed {obj} in {drawer} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # Try to pull each drawer handle (execute_pull)
        for drawer in drawer_names:
            for handle in handle_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying execute_pull {drawer} with {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Pulled {drawer} with {handle} at {loc}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")

        # Try to push each drawer (execute_push)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_push {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Pushed {drawer} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # Try to sweep each object (execute_sweep)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying execute_sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Swept {obj} at {loc}")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Used gripper")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Finished trying all available skills. Check logs for failed preconditions or missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()