# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The missing predicate is likely related to the robot's location, specifically (robot-at drawer-area).
        # We need to explore and confirm the robot can be at 'drawer-area' and that this predicate is recognized.

        # For exploration, we will attempt to move the robot to 'drawer-area' and check if the predicate is set.
        # We assume 'drawer-area' is a valid location in the environment.

        # Get all locations from positions (assuming keys like 'drawer-area', 'table-area', etc.)
        location_names = [k for k in positions.keys() if 'area' in k or 'location' in k]
        if 'drawer-area' in positions:
            drawer_area = 'drawer-area'
        else:
            # Fallback: try to find a location that matches 'drawer'
            drawer_area = None
            for loc in location_names:
                if 'drawer' in loc:
                    drawer_area = loc
                    break

        if drawer_area is None:
            print("[Exploration] Could not find 'drawer-area' in object positions. Exploration aborted.")
            return

        # Try to get the robot's current location
        # This may be available in the observation or positions; otherwise, we assume a default
        robot_location = None
        for loc in location_names:
            if 'robot' in loc:
                robot_location = loc
                break
        if robot_location is None:
            # Fallback: pick the first location as the starting point
            robot_location = location_names[0]

        print(f"[Exploration] Robot initial location: {robot_location}")
        print(f"[Exploration] Target location for exploration: {drawer_area}")

        # Use the available skill 'execute_go' to move the robot to 'drawer-area'
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Executed execute_go from {robot_location} to {drawer_area}.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")
            return

        # After moving, check if the predicate (robot-at drawer-area) is now true.
        # This would typically be checked via the environment's state or observation.
        # For demonstration, we print a message.
        print("[Exploration] Please verify if (robot-at drawer-area) is now true in the environment state.")

        # === End of Exploration Phase ===

        # === Main Task Plan Execution ===
        # Here you would continue with the oracle plan, using only the available skills.
        # For example:
        # obs, reward, done = execute_pick(env, task, obj_name, location, ...)
        # obs, reward, done = execute_place(env, task, obj_name, drawer_name, location, ...)
        # etc.
        # Since the main plan is not specified, we stop after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()