# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The predicate (robot-at floor) is missing or not handled.
        # We will attempt to use available skills to explore and identify the missing predicate.

        # For demonstration, let's assume 'floor' is a location in the environment.
        # We'll try to move the robot to 'floor' using the available skills and observe the result.

        # Find the robot's current location and the 'floor' location
        robot_location = None
        floor_location = None

        # Try to infer robot and floor from positions dictionary
        for name, pos in positions.items():
            if 'robot' in name or 'robby' in name:
                robot_location = name
            if 'floor' in name:
                floor_location = name

        # If not found, use default names
        if robot_location is None:
            robot_location = 'robot'
        if floor_location is None:
            floor_location = 'floor'

        print(f"[Exploration] Robot location: {robot_location}, Floor location: {floor_location}")

        # Try to move the robot to the floor using execute_go (if available)
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {floor_location} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=floor_location
            )
            print(f"[Exploration] execute_go result: done={done}")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # If execute_go is not sufficient, try other skills to explore the environment
        # For example, try to pick or sweep at the floor location
        try:
            print(f"[Exploration] Attempting to sweep at {floor_location} using execute_sweep.")
            obs, reward, done = execute_sweep(
                env,
                task,
                object_name=floor_location,
                location=floor_location
            )
            print(f"[Exploration] execute_sweep result: done={done}")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # Try to pick an object at the floor location if any object is present
        for obj_name in positions:
            if 'object' in obj_name or 'ball' in obj_name:
                try:
                    print(f"[Exploration] Attempting to pick {obj_name} at {floor_location} using execute_pick.")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        object_name=obj_name,
                        location=floor_location
                    )
                    print(f"[Exploration] execute_pick result: done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj_name}: {e}")

        # The above exploration attempts to trigger any missing predicate related to (robot-at floor)
        # and to discover if the environment responds to these actions.

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan Execution ===
        # Here you would continue with the actual oracle plan steps, using only the available skills.
        # For this task, the focus is on the exploration phase to identify the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()