# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check for (room-bright) predicate.
        # We'll attempt to execute actions that require (room-bright) and handle failure if it's missing.

        # For demonstration, let's try to execute a skill that requires (room-bright).
        # We'll use 'execute_pick' as an example, since it requires (room-bright) in its precondition.

        # Find an object on the floor and its location
        object_to_pick = None
        pick_location = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('on_floor', False):
                object_to_pick = obj_name
                pick_location = obj_info.get('location', None)
                break

        # If not found, just pick the first object and its location
        if object_to_pick is None:
            for obj_name, obj_info in positions.items():
                object_to_pick = obj_name
                pick_location = obj_info.get('location', None)
                break

        # Find a switch and its location for execute_push_switch
        switch_name = None
        switch_location = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '') == 'switch':
                switch_name = obj_name
                switch_location = obj_info.get('location', None)
                break

        # Find the robot's current location
        robot_location = None
        for obj_name, obj_info in positions.items():
            if obj_name == 'robot':
                robot_location = obj_info.get('location', None)
                break

        # If not found, try to infer from any location
        if robot_location is None:
            for obj_name, obj_info in positions.items():
                if obj_info.get('is_robot', False):
                    robot_location = obj_info.get('location', None)
                    break

        # === Try to pick the object (should fail if room is not bright) ===
        pick_success = False
        try:
            if object_to_pick is not None and pick_location is not None:
                print(f"[Exploration] Attempting to pick {object_to_pick} at {pick_location} (should require room-bright).")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=object_to_pick,
                    p=pick_location
                )
                pick_success = True
                print("[Exploration] Pick succeeded. (room-bright) likely present.")
            else:
                print("[Exploration] No suitable object found for pick.")
        except Exception as e:
            print(f"[Exploration] Pick failed as expected (missing (room-bright)): {e}")
            pick_success = False

        # === If pick failed, try to turn on the light (execute_push_switch) ===
        if not pick_success:
            try:
                if switch_name is not None and switch_location is not None:
                    print(f"[Exploration] Attempting to push switch {switch_name} at {switch_location} to turn on light.")
                    obs, reward, done = execute_push_switch(
                        env,
                        task,
                        s=switch_name,
                        p=switch_location
                    )
                    print("[Exploration] Switch pushed. Room should now be bright.")
                else:
                    print("[Exploration] No switch found in environment.")
            except Exception as e:
                print(f"[Exploration] Failed to push switch: {e}")

            # Try pick again after turning on the light
            try:
                if object_to_pick is not None and pick_location is not None:
                    print(f"[Exploration] Retrying pick {object_to_pick} at {pick_location} after turning on light.")
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=object_to_pick,
                        p=pick_location
                    )
                    pick_success = True
                    print("[Exploration] Pick succeeded after turning on light. (room-bright) predicate was missing and is now set.")
                else:
                    print("[Exploration] No suitable object found for pick.")
            except Exception as e:
                print(f"[Exploration] Pick still failed: {e}")

        # === Continue with the rest of the oracle plan as needed ===
        # For demonstration, we can attempt to place the object in a drawer if available

        # Find a drawer and its location
        drawer_name = None
        drawer_location = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '') == 'drawer':
                drawer_name = obj_name
                drawer_location = obj_info.get('location', None)
                break

        if pick_success and drawer_name is not None and drawer_location is not None:
            try:
                print(f"[Task] Attempting to place {object_to_pick} in {drawer_name} at {drawer_location}.")
                obs, reward, done = execute_place(
                    env,
                    task,
                    o=object_to_pick,
                    d=drawer_name,
                    p=drawer_location
                )
                print("[Task] Place succeeded.")
            except Exception as e:
                print(f"[Task] Place failed: {e}")
        else:
            print("[Task] Skipping place action (pick or drawer not available).")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()