# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check the state of the drawer (drawer-open drawer_bottom)
        # and explore to determine if any predicate is missing (e.g., lock-known, drawer-locked, etc.)

        # For demonstration, let's assume the following object and location names
        # (You should replace these with actual names from your environment/positions)
        robot_location = None
        drawer_name = "drawer_bottom"
        handle_name = "handle_bottom"
        switch_name = "switch_1"
        object_to_pick = "object_1"
        place_location = "drawer_bottom"
        robot_start_location = None

        # Try to infer actual names from positions if possible
        for name in positions:
            if "drawer" in name and "bottom" in name:
                drawer_name = name
            if "handle" in name and "bottom" in name:
                handle_name = name
            if "switch" in name:
                switch_name = name
            if "object" in name:
                object_to_pick = name
            if "robot" in name or "start" in name:
                robot_start_location = name

        # Fallbacks if not found
        if not robot_start_location:
            # Try to get any location
            for name in positions:
                if "location" in name or "room" in name:
                    robot_start_location = name
                    break

        # 1. Exploration: Try to pull the drawer to check lock-known predicate
        print("[Exploration] Attempting to pull the drawer to check lock-known predicate.")
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                positions[drawer_name] if drawer_name in positions else None
            )
            print("[Exploration] Pull action executed. Checking for lock-known predicate.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_pull: {e}")

        # 2. Exploration: Try to open the drawer (if not already open)
        print("[Exploration] Attempting to open the drawer if not already open.")
        try:
            # If the drawer is not open, try to open it
            # This may require the robot to hold the handle first
            # For simplicity, try to pick the handle and then pull
            obs, reward, done = execute_pick(
                env,
                task,
                handle_name,
                positions[handle_name] if handle_name in positions else None
            )
            print("[Exploration] Picked handle. Now trying to pull the drawer.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                positions[drawer_name] if drawer_name in positions else None
            )
            print("[Exploration] Drawer pull attempted.")
        except Exception as e:
            print(f"[Exploration] Exception during drawer open: {e}")

        # 3. Exploration: Try to push the switch to turn on the light if room is dark
        print("[Exploration] Attempting to push the switch to turn on the light.")
        try:
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                positions[switch_name] if switch_name in positions else None
            )
            print("[Exploration] Switch push attempted.")
        except Exception as e:
            print(f"[Exploration] Exception during switch push: {e}")

        # 4. Exploration: Try to pick an object from the floor
        print("[Exploration] Attempting to pick an object from the floor.")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_to_pick,
                positions[object_to_pick] if object_to_pick in positions else None
            )
            print("[Exploration] Picked object from floor.")
        except Exception as e:
            print(f"[Exploration] Exception during object pick: {e}")

        # 5. Exploration: Try to place the object in the drawer
        print("[Exploration] Attempting to place the object in the drawer.")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_to_pick,
                drawer_name,
                positions[drawer_name] if drawer_name in positions else None
            )
            print("[Exploration] Placed object in drawer.")
        except Exception as e:
            print(f"[Exploration] Exception during object place: {e}")

        # 6. Exploration: Try to push the drawer closed
        print("[Exploration] Attempting to push the drawer closed.")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name,
                positions[drawer_name] if drawer_name in positions else None
            )
            print("[Exploration] Drawer push attempted.")
        except Exception as e:
            print(f"[Exploration] Exception during drawer push: {e}")

        # 7. Exploration: Try to move the robot to another location (if available)
        print("[Exploration] Attempting to move the robot to another location.")
        try:
            # Find another location to move to
            other_location = None
            for name in positions:
                if name != robot_start_location and ("location" in name or "room" in name):
                    other_location = name
                    break
            if other_location and robot_start_location:
                obs, reward, done = execute_go(
                    env,
                    task,
                    robot_start_location,
                    other_location
                )
                print(f"[Exploration] Robot moved from {robot_start_location} to {other_location}.")
            else:
                print("[Exploration] No alternative location found for movement.")
        except Exception as e:
            print(f"[Exploration] Exception during robot move: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicates or failed actions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
