# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We attempt to use available exploration skills to discover missing predicates.
        # The exploration domain provides actions like execute_go_identify, execute_go_temperature, etc.
        # However, in this environment, we only use the provided skills.

        # Since the available skills do not include explicit exploration actions,
        # we simulate an exploration phase by attempting to execute each available skill
        # and observing the environment for changes or errors, which may indicate missing predicates.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # Map skill names to actual functions (assuming skill_code provides them)
        skill_func_map = {
            'execute_pick': execute_pick if 'execute_pick' in globals() else None,
            'execute_place': execute_place if 'execute_place' in globals() else None,
            'execute_push': execute_push if 'execute_push' in globals() else None,
            'execute_pull': execute_pull if 'execute_pull' in globals() else None,
            'execute_sweep': execute_sweep if 'execute_sweep' in globals() else None,
            'execute_rotate': execute_rotate if 'execute_rotate' in globals() else None,
            'execute_go': execute_go if 'execute_go' in globals() else None,
            'execute_gripper': execute_gripper if 'execute_gripper' in globals() else None,
            # The last entry is a mapping, handle accordingly
            'execute_push_swtich -> push': execute_push_switch if 'execute_push_switch' in globals() else None
        }

        # Attempt to execute each skill with dummy or inferred arguments to probe for missing predicates
        # This is a safe exploration: catch exceptions and log them
        print("[Exploration] Starting exploration phase to identify missing predicates...")
        for skill_name, skill_func in skill_func_map.items():
            if skill_func is None:
                print(f"[Exploration] Skill {skill_name} not implemented in skill_code.")
                continue
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                # Prepare dummy arguments based on skill signature
                # For demonstration, use the first available object/location from positions
                # In a real scenario, you would parse the observation/init for correct arguments
                args = []
                if skill_name == 'execute_pick':
                    # Find an object on the floor and a location
                    obj = next((k for k in positions if 'object' in k), None)
                    loc = next((k for k in positions if 'location' in k), None)
                    if obj and loc:
                        args = [env, task, obj, loc]
                elif skill_name == 'execute_place':
                    # Find an object, a drawer, and a location
                    obj = next((k for k in positions if 'object' in k), None)
                    drawer = next((k for k in positions if 'drawer' in k), None)
                    loc = next((k for k in positions if 'location' in k), None)
                    if obj and drawer and loc:
                        args = [env, task, obj, drawer, loc]
                elif skill_name == 'execute_push':
                    drawer = next((k for k in positions if 'drawer' in k), None)
                    loc = next((k for k in positions if 'location' in k), None)
                    if drawer and loc:
                        args = [env, task, drawer, loc]
                elif skill_name == 'execute_pull':
                    drawer = next((k for k in positions if 'drawer' in k), None)
                    handle = next((k for k in positions if 'handle' in k), None)
                    loc = next((k for k in positions if 'location' in k), None)
                    if drawer and handle and loc:
                        args = [env, task, drawer, handle, loc]
                elif skill_name == 'execute_sweep':
                    obj = next((k for k in positions if 'object' in k), None)
                    loc = next((k for k in positions if 'location' in k), None)
                    if obj and loc:
                        args = [env, task, obj, loc]
                elif skill_name == 'execute_rotate':
                    # No parameters in domain, so just env, task
                    args = [env, task]
                elif skill_name == 'execute_go':
                    from_loc = next((k for k in positions if 'location' in k), None)
                    to_loc = next((k for k in positions if 'location' in k and k != from_loc), None)
                    if from_loc and to_loc:
                        args = [env, task, from_loc, to_loc]
                elif skill_name == 'execute_gripper':
                    args = [env, task]
                elif skill_name == 'execute_push_swtich -> push':
                    switch = next((k for k in positions if 'switch' in k), None)
                    loc = next((k for k in positions if 'location' in k), None)
                    if switch and loc:
                        args = [env, task, switch, loc]
                # Only call if we have enough arguments
                if args:
                    obs, reward, done = skill_func(*args)
                    print(f"[Exploration] Skill {skill_name} executed successfully.")
                else:
                    print(f"[Exploration] Not enough arguments for {skill_name}, skipping.")
            except Exception as e:
                print(f"[Exploration] Exception during {skill_name}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the main plan (oracle plan).
        # For demonstration, we show a generic plan execution loop.
        # In practice, you would parse and execute the oracle plan step-by-step.

        # Example: Suppose the plan is to turn on the light, pick an object, open a drawer, and place the object.
        # The following is a generic sequence using available skills and positions.

        try:
            # 1. Turn on the light if needed (execute_push_switch)
            switch = next((k for k in positions if 'switch' in k), None)
            loc = next((k for k in positions if 'location' in k), None)
            if switch and loc and 'execute_push_switch' in globals():
                print("[Task] Turning on the light...")
                obs, reward, done = execute_push_switch(env, task, switch, loc)

            # 2. Pick up an object
            obj = next((k for k in positions if 'object' in k), None)
            if obj and loc and 'execute_pick' in globals():
                print(f"[Task] Picking up object: {obj}")
                obs, reward, done = execute_pick(env, task, obj, loc)

            # 3. Open a drawer (pull)
            drawer = next((k for k in positions if 'drawer' in k), None)
            handle = next((k for k in positions if 'handle' in k), None)
            if drawer and handle and loc and 'execute_pull' in globals():
                print(f"[Task] Opening drawer: {drawer} with handle: {handle}")
                obs, reward, done = execute_pull(env, task, drawer, handle, loc)

            # 4. Place the object in the drawer
            if obj and drawer and loc and 'execute_place' in globals():
                print(f"[Task] Placing object: {obj} in drawer: {drawer}")
                obs, reward, done = execute_place(env, task, obj, drawer, loc)

            # 5. Close the drawer (push)
            if drawer and loc and 'execute_push' in globals():
                print(f"[Task] Closing drawer: {drawer}")
                obs, reward, done = execute_push(env, task, drawer, loc)

            print("[Task] Task plan executed successfully.")

        except Exception as e:
            print(f"[Task] Exception during plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
