# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try available skills to explore the environment and observe which predicates are required.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to execute each skill in a safe, exploratory way.
        # This will help us identify which predicates are missing or required for the domain.

        # Get a list of objects, drawers, locations, and switches from positions (if available)
        # These keys should match your environment's naming conventions
        objects = [k for k in positions if 'object' in k]
        drawers = [k for k in positions if 'drawer' in k]
        handles = [k for k in positions if 'handle' in k]
        locations = [k for k in positions if 'location' in k or 'room' in k]
        switches = [k for k in positions if 'switch' in k or 'light' in k]

        # Fallback: If no explicit locations, use all unique position values as locations
        if not locations:
            locations = list(set([v for v in positions.values()]))

        # Try to get the robot's current location
        robot_location = None
        for k in positions:
            if 'robot' in k or 'robby' in k:
                robot_location = positions[k]
                break

        # 1. Try to turn on the light if a switch is present
        if switches and locations:
            try:
                print("[Exploration] Trying to execute_push_switch (turn on light)")
                # Assume first switch and first location
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=switches[0],
                    location_name=locations[0]
                )
                print("[Exploration] execute_push_switch succeeded.")
            except Exception as e:
                print("[Exploration] execute_push_switch failed:", e)

        # 2. Try to pick up an object
        if objects and locations:
            try:
                print("[Exploration] Trying to execute_pick (pick up object)")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=objects[0],
                    location_name=locations[0]
                )
                print("[Exploration] execute_pick succeeded.")
            except Exception as e:
                print("[Exploration] execute_pick failed:", e)

        # 3. Try to open a drawer (pull)
        if drawers and handles and locations:
            try:
                print("[Exploration] Trying to execute_pull (open drawer)")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name=drawers[0],
                    handle_name=handles[0],
                    location_name=locations[0]
                )
                print("[Exploration] execute_pull succeeded.")
            except Exception as e:
                print("[Exploration] execute_pull failed:", e)

        # 4. Try to place an object in a drawer
        if objects and drawers and locations:
            try:
                print("[Exploration] Trying to execute_place (place object in drawer)")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_name=objects[0],
                    drawer_name=drawers[0],
                    location_name=locations[0]
                )
                print("[Exploration] execute_place succeeded.")
            except Exception as e:
                print("[Exploration] execute_place failed:", e)

        # 5. Try to push (close) a drawer
        if drawers and locations:
            try:
                print("[Exploration] Trying to execute_push (close drawer)")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawers[0],
                    location_name=locations[0]
                )
                print("[Exploration] execute_push succeeded.")
            except Exception as e:
                print("[Exploration] execute_push failed:", e)

        # 6. Try to move (go) between locations
        if len(locations) >= 2:
            try:
                print("[Exploration] Trying to execute_go (move between locations)")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=locations[0],
                    to_location=locations[1]
                )
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print("[Exploration] execute_go failed:", e)

        # 7. Try to sweep an object
        if objects and locations:
            try:
                print("[Exploration] Trying to execute_sweep (sweep object)")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    object_name=objects[0],
                    location_name=locations[0]
                )
                print("[Exploration] execute_sweep succeeded.")
            except Exception as e:
                print("[Exploration] execute_sweep failed:", e)

        # 8. Try to use gripper
        try:
            print("[Exploration] Trying to execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        print("[Exploration] Exploration phase complete. Check logs for missing predicate errors.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
