# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will attempt to use available skills to explore the environment and infer which predicate is missing.
        # Since we cannot define new skills, we use only the provided ones.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # For exploration, we will try to execute each skill in a safe, diagnostic way
        # and observe the environment's response to infer missing predicates.

        # 1. Try to move (execute_go) between locations if possible
        # 2. Try to pick and place objects, open/close drawers, etc.
        # 3. If an action fails, log the error and continue.
        # 4. After each action, observe the state and look for clues about missing predicates.

        # For demonstration, we will attempt to:
        # - Turn on the light if the room is dark (execute_push_switch)
        # - Pick up an object from the floor (execute_pick)
        # - Open a drawer (execute_pull)
        # - Place the object in the drawer (execute_place)
        # - Close the drawer (execute_push)
        # - Move between locations (execute_go)
        # - Sweep the floor (execute_sweep)
        # - Use the gripper (execute_gripper)

        # The actual object and location names should be replaced with those from the environment.
        # We'll use placeholder names and handle missing keys gracefully.

        # Helper function to get a key by partial match
        def get_key_by_partial(d, partial):
            for k in d:
                if partial in k:
                    return k
            return None

        # Try to find a switch, an object, a drawer, and two locations
        switch_name = get_key_by_partial(positions, 'switch')
        object_name = get_key_by_partial(positions, 'object')
        drawer_name = get_key_by_partial(positions, 'drawer')
        handle_name = get_key_by_partial(positions, 'handle')
        location_names = [k for k in positions if 'location' in k or 'room' in k or 'area' in k]
        if len(location_names) < 2:
            # Fallback: use any two keys
            location_names = list(positions.keys())[:2]

        # 1. Try to turn on the light (execute_push_switch)
        try:
            if switch_name and location_names:
                print(f"[Exploration] Trying to turn on the light using switch '{switch_name}' at location '{location_names[0]}'")
                obs, reward, done = execute_push_switch(
                    env, task, switch_name, location_names[0]
                )
        except Exception as e:
            print(f"[Exploration] execute_push_switch failed: {e}")

        # 2. Try to pick up an object from the floor (execute_pick)
        try:
            if object_name and location_names:
                print(f"[Exploration] Trying to pick up object '{object_name}' at location '{location_names[0]}'")
                obs, reward, done = execute_pick(
                    env, task, object_name, location_names[0]
                )
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to open a drawer (execute_pull)
        try:
            if drawer_name and handle_name and location_names:
                print(f"[Exploration] Trying to open drawer '{drawer_name}' using handle '{handle_name}' at location '{location_names[0]}'")
                obs, reward, done = execute_pull(
                    env, task, drawer_name, handle_name, location_names[0]
                )
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try to place the object in the drawer (execute_place)
        try:
            if object_name and drawer_name and location_names:
                print(f"[Exploration] Trying to place object '{object_name}' in drawer '{drawer_name}' at location '{location_names[0]}'")
                obs, reward, done = execute_place(
                    env, task, object_name, drawer_name, location_names[0]
                )
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")

        # 5. Try to close the drawer (execute_push)
        try:
            if drawer_name and location_names:
                print(f"[Exploration] Trying to close drawer '{drawer_name}' at location '{location_names[0]}'")
                obs, reward, done = execute_push(
                    env, task, drawer_name, location_names[0]
                )
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to move between two locations (execute_go)
        try:
            if len(location_names) >= 2:
                print(f"[Exploration] Trying to move from '{location_names[0]}' to '{location_names[1]}'")
                obs, reward, done = execute_go(
                    env, task, location_names[0], location_names[1]
                )
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # 7. Try to sweep the floor (execute_sweep)
        try:
            if object_name and location_names:
                print(f"[Exploration] Trying to sweep object '{object_name}' at location '{location_names[0]}'")
                obs, reward, done = execute_sweep(
                    env, task, object_name, location_names[0]
                )
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # 8. Try to use the gripper (execute_gripper)
        try:
            print(f"[Exploration] Trying to use the gripper")
            obs, reward, done = execute_gripper(
                env, task
            )
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # After exploration, print a message to indicate that the missing predicate may have been identified
        print("[Exploration] Exploration phase complete. Check logs for failed actions and missing predicates.")

        # === End of Exploration ===

        # TODO: After identifying the missing predicate, you would update the domain/problem files accordingly
        # and re-run the planner. This is outside the scope of this code, which focuses on exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()