# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (room-bright) is missing or not satisfied.
        # According to the domain, (room-bright) is achieved by execute_push_switch (turning on the light).
        # So, before any action that requires (room-bright), we must ensure the light is on.

        # Try to execute execute_push_switch if the room is dark.
        # We assume the environment provides a way to check room brightness in obs or descriptions.
        # If not, we attempt to turn on the light as a safe default.

        # Find the switch and its location from positions or descriptions
        switch_name = None
        switch_location = None
        for obj_name, pos in positions.items():
            if 'switch' in obj_name.lower():
                switch_name = obj_name
                switch_location = pos
                break

        # If switch is not found, try to get from descriptions or fallback
        if switch_name is None:
            # Fallback: try to find from descriptions if available
            for desc in descriptions:
                if 'switch' in desc.lower():
                    switch_name = desc
                    break

        # Assume robot's current location is available in obs or positions
        robot_location = None
        for obj_name, pos in positions.items():
            if 'robot' in obj_name.lower():
                robot_location = pos
                break

        # If not found, fallback to a default location
        if robot_location is None:
            robot_location = (0, 0, 0)

        # Try to turn on the light if needed
        try:
            # Check if room is dark (if such info is available)
            room_bright = False
            if 'room-bright' in obs:
                room_bright = obs['room-bright']
            elif 'room-bright' in descriptions:
                room_bright = True
            else:
                # If not available, assume it's dark and try to turn on
                room_bright = False

            if not room_bright:
                print("[Exploration] Room is dark. Attempting to turn on the light using execute_push_switch.")
                if switch_name is not None and switch_location is not None:
                    obs, reward, done = execute_push_switch(
                        env,
                        task,
                        switch_name,
                        switch_location
                    )
                    print("[Exploration] Light switch action executed.")
                else:
                    print("[Exploration] Switch not found. Skipping light switch action.")
            else:
                print("[Exploration] Room is already bright. No need to turn on the light.")
        except Exception as e:
            print(f"[Exploration] Exception during light switch action: {e}")

        # === Main Task Plan ===
        # At this point, the room should be bright and we can proceed with the oracle plan.
        # The following is a generic example of how to proceed with the available skills.
        # You should replace these with the actual plan steps for your task.

        # Example: Pick up an object from the floor and place it in a drawer
        # Find an object on the floor and a drawer
        object_on_floor = None
        object_location = None
        drawer_name = None
        drawer_location = None
        handle_name = None

        for obj_name, pos in positions.items():
            if 'object' in obj_name.lower() or 'ball' in obj_name.lower():
                object_on_floor = obj_name
                object_location = pos
                break

        for obj_name, pos in positions.items():
            if 'drawer' in obj_name.lower():
                drawer_name = obj_name
                drawer_location = pos
                break

        for obj_name, pos in positions.items():
            if 'handle' in obj_name.lower():
                handle_name = obj_name
                break

        # Move to the object's location if not already there
        try:
            if object_location is not None and robot_location != object_location:
                print(f"[Task] Moving to object {object_on_floor} at {object_location}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    robot_location,
                    object_location
                )
                robot_location = object_location
        except Exception as e:
            print(f"[Task] Exception during move to object: {e}")

        # Pick up the object
        try:
            if object_on_floor is not None and object_location is not None:
                print(f"[Task] Picking up object {object_on_floor} at {object_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_on_floor,
                    object_location
                )
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")

        # Move to the drawer location
        try:
            if drawer_location is not None and robot_location != drawer_location:
                print(f"[Task] Moving to drawer {drawer_name} at {drawer_location}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    robot_location,
                    drawer_location
                )
                robot_location = drawer_location
        except Exception as e:
            print(f"[Task] Exception during move to drawer: {e}")

        # Open the drawer (if handle and drawer are known)
        try:
            if handle_name is not None and drawer_name is not None and drawer_location is not None:
                print(f"[Task] Pulling handle {handle_name} to open drawer {drawer_name}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name,
                    handle_name,
                    drawer_location
                )
        except Exception as e:
            print(f"[Task] Exception during drawer pull: {e}")

        # Place the object in the drawer
        try:
            if object_on_floor is not None and drawer_name is not None and drawer_location is not None:
                print(f"[Task] Placing object {object_on_floor} in drawer {drawer_name}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    object_on_floor,
                    drawer_name,
                    drawer_location
                )
        except Exception as e:
            print(f"[Task] Exception during place: {e}")

        # Optionally, close the drawer
        try:
            if drawer_name is not None and drawer_location is not None:
                print(f"[Task] Pushing drawer {drawer_name} closed")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name,
                    drawer_location
                )
        except Exception as e:
            print(f"[Task] Exception during drawer push: {e}")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()