# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the robot's location and update our knowledge.

        # For demonstration, let's assume the robot starts at 'floor' location.
        # We'll use the available skills to explore and confirm the robot's position.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper', 'execute_push_swtich -> push']

        # Since the exploration domain provides actions like execute_go_identify, but our available skills do not include this,
        # we will use 'execute_go' to move and 'execute_pick' to interact, as per the primitive_skills_static_lock_v2 domain.

        # Let's try to use 'execute_go' to move the robot and check if the robot is at the expected location.
        # We'll also print out the robot's location to confirm.

        # For this example, let's assume the environment has locations: 'floor', 'drawer_area', 'switch_area'
        # and objects: 'obj1', 'drawer1', 'switch1'
        # These names should be replaced with actual names from your environment if available.

        # Example object and location names (replace with actual names if different)
        robot_location = 'floor'
        drawer_location = 'drawer_area'
        switch_location = 'switch_area'
        object_on_floor = None
        drawer_name = None
        switch_name = None

        # Try to infer object and location names from positions dictionary
        for name in positions:
            if 'drawer' in name:
                drawer_name = name
            elif 'switch' in name:
                switch_name = name
            elif 'obj' in name or 'ball' in name or 'cube' in name:
                object_on_floor = name

        # Fallbacks if not found
        if object_on_floor is None:
            object_on_floor = 'obj1'
        if drawer_name is None:
            drawer_name = 'drawer1'
        if switch_name is None:
            switch_name = 'switch1'

        # Print out what we found
        print(f"[Exploration] Robot assumed at location: {robot_location}")
        print(f"[Exploration] Object on floor: {object_on_floor}")
        print(f"[Exploration] Drawer: {drawer_name}")
        print(f"[Exploration] Switch: {switch_name}")

        # Step 1: If the room is dark, turn on the light using execute_push_switch (if available)
        # The skill list has 'execute_push_swtich -> push', so we try to use 'execute_push_switch' if defined
        try:
            # Try to turn on the light if the room is dark
            print("[Exploration] Attempting to turn on the light (if needed)...")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name,
                switch_location
            )
            print("[Exploration] Light switch action executed.")
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_push_switch': {e}")

        # Step 2: Move the robot to the drawer area using execute_go
        try:
            print(f"[Exploration] Moving robot from {robot_location} to {drawer_location}...")
            obs, reward, done = execute_go(
                env,
                task,
                robot_location,
                drawer_location
            )
            print(f"[Exploration] Robot moved to {drawer_location}.")
            robot_location = drawer_location
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_go': {e}")

        # Step 3: Pick up the object from the floor (if present) using execute_pick
        try:
            print(f"[Exploration] Attempting to pick up {object_on_floor} at {robot_location}...")
            obs, reward, done = execute_pick(
                env,
                task,
                object_on_floor,
                robot_location
            )
            print(f"[Exploration] Picked up {object_on_floor}.")
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_pick': {e}")

        # Step 4: Open the drawer (if needed) using execute_pull
        # For execute_pull, we need the handle object (assume it's the same as drawer_name for this example)
        try:
            print(f"[Exploration] Attempting to open drawer {drawer_name} with handle {drawer_name} at {robot_location}...")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                drawer_name,
                robot_location
            )
            print(f"[Exploration] Drawer {drawer_name} opened.")
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_pull': {e}")

        # Step 5: Place the object into the drawer using execute_place
        try:
            print(f"[Exploration] Attempting to place {object_on_floor} into {drawer_name} at {robot_location}...")
            obs, reward, done = execute_place(
                env,
                task,
                object_on_floor,
                drawer_name,
                robot_location
            )
            print(f"[Exploration] Placed {object_on_floor} into {drawer_name}.")
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_place': {e}")

        # Step 6: Close the drawer using execute_push
        try:
            print(f"[Exploration] Attempting to close drawer {drawer_name} at {robot_location}...")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name,
                robot_location
            )
            print(f"[Exploration] Drawer {drawer_name} closed.")
        except Exception as e:
            print(f"[Exploration] Could not execute 'execute_push': {e}")

        # At this point, the exploration phase should have exercised all relevant predicates and skills,
        # and any missing predicate (such as (robot-at floor)) should have been discovered through feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()