# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the exploration domain knowledge to try to discover which predicate is missing.
        # This is done by attempting to execute exploration skills and observing the environment.

        # For demonstration, we attempt to identify, weigh, and check durability of all objects at all locations.
        # We also attempt to check lock status of objects that can be pulled.

        # Assume positions is a dict: {object_name: (x, y, z, location_name, type)}
        # We'll iterate over all objects and locations to perform exploration actions.

        print("[Exploration] Starting exploration phase to discover missing predicates...")

        # Collect all objects and locations from positions
        objects = []
        locations = set()
        for obj_name, info in positions.items():
            objects.append(obj_name)
            if len(info) >= 4:
                locations.add(info[3])
        locations = list(locations)

        # Try to move to each location and identify objects there
        for loc in locations:
            try:
                # Move robot to location (if not already there)
                # We assume a function execute_go exists in skill_code
                # For the first location, we assume robot is already there after reset
                print(f"[Exploration] Moving to location: {loc}")
                obs, reward, done = execute_go(env, task, from_location=None, to_location=loc)
            except Exception as e:
                print(f"[Exploration] Error during execute_go to {loc}: {e}")

            # Try to identify all objects at this location
            for obj_name, info in positions.items():
                if len(info) >= 4 and info[3] == loc:
                    try:
                        print(f"[Exploration] Attempting to identify object: {obj_name} at {loc}")
                        # Try to pick up the object to learn its weight and durability
                        obs, reward, done = execute_pick(env, task, obj_name, loc)
                        obs, reward, done = execute_gripper(env, task)
                        # Try to place the object back if possible
                        # (Assume a default drawer or location for place, if available)
                    except Exception as e:
                        print(f"[Exploration] Error during execute_pick for {obj_name} at {loc}: {e}")

        # Try to pull handles to check lock-known predicate
        for obj_name, info in positions.items():
            # If the object is a handle or can be pulled, try to pull it
            if 'handle' in obj_name or 'drawer' in obj_name:
                for loc in locations:
                    try:
                        print(f"[Exploration] Attempting to pull {obj_name} at {loc}")
                        obs, reward, done = execute_pull(env, task, obj_name, obj_name, loc)
                    except Exception as e:
                        print(f"[Exploration] Error during execute_pull for {obj_name} at {loc}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the main oracle plan using only predefined skills.
        # Example: Turn on the light if room is dark, pick up an object, open drawer, place object, etc.

        # 1. Try to turn on the light if needed
        try:
            print("[Task] Attempting to turn on the light (execute_push_switch)...")
            # Assume switch and location are known (replace with actual names from positions if available)
            switch_name = None
            switch_location = None
            for obj_name, info in positions.items():
                if 'switch' in obj_name:
                    switch_name = obj_name
                    if len(info) >= 4:
                        switch_location = info[3]
                    break
            if switch_name and switch_location:
                obs, reward, done = execute_push_switch(env, task, switch_name, switch_location)
            else:
                print("[Task] No switch found in positions; skipping light activation.")
        except Exception as e:
            print(f"[Task] Error during execute_push_switch: {e}")

        # 2. Pick up an object from the floor
        try:
            print("[Task] Attempting to pick up an object from the floor...")
            for obj_name, info in positions.items():
                if 'object' in info or 'obj' in obj_name or 'ball' in obj_name:
                    if len(info) >= 4:
                        obj_location = info[3]
                        obs, reward, done = execute_pick(env, task, obj_name, obj_location)
                        break
        except Exception as e:
            print(f"[Task] Error during execute_pick: {e}")

        # 3. Open a drawer (if needed)
        try:
            print("[Task] Attempting to open a drawer...")
            drawer_name = None
            handle_name = None
            drawer_location = None
            for obj_name, info in positions.items():
                if 'drawer' in obj_name:
                    drawer_name = obj_name
                    if len(info) >= 4:
                        drawer_location = info[3]
                if 'handle' in obj_name:
                    handle_name = obj_name
            if drawer_name and handle_name and drawer_location:
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
            else:
                print("[Task] No drawer/handle found in positions; skipping drawer open.")
        except Exception as e:
            print(f"[Task] Error during execute_pull: {e}")

        # 4. Place the object in the drawer
        try:
            print("[Task] Attempting to place object in the drawer...")
            if drawer_name and obj_name and drawer_location:
                obs, reward, done = execute_place(env, task, obj_name, drawer_name, drawer_location)
            else:
                print("[Task] Missing object or drawer for place; skipping.")
        except Exception as e:
            print(f"[Task] Error during execute_place: {e}")

        # 5. Close the drawer (if needed)
        try:
            print("[Task] Attempting to close the drawer...")
            if drawer_name and drawer_location:
                obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            else:
                print("[Task] No drawer found for push; skipping.")
        except Exception as e:
            print(f"[Task] Error during execute_push: {e}")

        print("[Task] Main task plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()