# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will perform an exploration phase using available skills to probe the environment
        # and try to identify which predicate is missing or required for the plan.

        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to gather such information.

        # Since we do not have the actual oracle plan, we will perform a systematic exploration
        # using the available skills and print out the results to help identify missing predicates.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # We will also catch exceptions to avoid crashing if a skill is not applicable.

        # 1. Try to move the robot to each location (if multiple locations exist)
        locations = [k for k, v in positions.items() if 'location' in k or 'room' in k or 'drawer' in k]
        objects = [k for k, v in positions.items() if 'object' in k or 'ball' in k or 'handle' in k or 'switch' in k]
        drawers = [k for k, v in positions.items() if 'drawer' in k]
        switches = [k for k, v in positions.items() if 'switch' in k]

        # Fallback: If no locations found, try to use all keys as possible locations
        if not locations:
            locations = list(positions.keys())

        # 2. Exploration: Try to use each skill and observe the effect
        print("[Exploration] Starting exploration phase to identify missing predicates or required knowledge.")

        # Try to move to each location (execute_go)
        for i in range(len(locations)):
            for j in range(len(locations)):
                if i == j:
                    continue
                from_loc = locations[i]
                to_loc = locations[j]
                try:
                    print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                    if done:
                        print(f"[Exploration] Moved from {from_loc} to {to_loc}.")
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {e}")

        # Try to pick up each object at each location (execute_pick)
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_pick for {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Picked {obj} at {loc}.")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # Try to place each object in each drawer at each location (execute_place)
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_place for {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Placed {obj} in {drawer} at {loc}.")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # Try to push and pull each drawer at each location (execute_push, execute_pull)
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_push for {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Pushed {drawer} at {loc}.")
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")
                # For pull, need a handle object; try all objects as handles
                for handle in objects:
                    try:
                        print(f"[Exploration] Trying execute_pull for {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        if done:
                            print(f"[Exploration] Pulled {drawer} with handle {handle} at {loc}.")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")

        # Try to sweep each object at each location (execute_sweep)
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_sweep for {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Swept {obj} at {loc}.")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # Try to rotate (if available)
        try:
            print(f"[Exploration] Trying execute_rotate")
            obs, reward, done = execute_rotate(env, task)
            if done:
                print(f"[Exploration] Rotated.")
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        # Try to use gripper (if available)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Used gripper.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # Try to push switch (if available)
        for switch in switches:
            for loc in locations:
                try:
                    print(f"[Exploration] Trying execute_push_switch for {switch} at {loc}")
                    obs, reward, done = execute_push_switch(env, task, switch, loc)
                    if done:
                        print(f"[Exploration] Pushed switch {switch} at {loc}.")
                except Exception as e:
                    print(f"[Exploration] execute_push_switch failed: {e}")

        print("[Exploration] Exploration phase complete. Check above logs for clues about missing predicates or required knowledge.")

        # === End of Exploration ===

        # TODO: After identifying the missing predicate, update the plan or initial state accordingly.
        # For now, the code performs a systematic exploration to help diagnose the planning failure.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
