# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try each available skill and observe the environment for changes,
        # to infer which predicate is missing or required for the plan to proceed.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to execute each skill with plausible arguments,
        # catch exceptions, and print the result. This helps to identify which skill (and thus
        # which predicate) is not applicable due to missing preconditions.

        # For this exploration, we assume the following:
        # - There is at least one object, one drawer, one location, and possibly a switch.
        # - The positions dictionary contains keys for objects, drawers, locations, and switches.

        # Extract object, drawer, location, and switch names from positions (if available)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name]
        switch_names = [name for name in positions if 'switch' in name]

        # Fallbacks in case names are not found
        if not object_names:
            object_names = list(positions.keys())
        if not drawer_names:
            drawer_names = list(positions.keys())
        if not location_names:
            location_names = list(positions.keys())
        if not switch_names:
            switch_names = list(positions.keys())

        # Select first available for each type
        obj = object_names[0] if object_names else None
        drawer = drawer_names[0] if drawer_names else None
        location = location_names[0] if location_names else None
        switch = switch_names[0] if switch_names else None

        # Try each skill and observe the result
        for skill in available_skills:
            print(f"[Exploration] Attempting skill: {skill}")
            try:
                if skill == 'execute_pick':
                    # Try picking up an object from a location
                    if obj and location:
                        obs, reward, done = execute_pick(env, task, obj, location)
                        print(f"Executed {skill}({obj}, {location})")
                elif skill == 'execute_place':
                    # Try placing an object into a drawer at a location
                    if obj and drawer and location:
                        obs, reward, done = execute_place(env, task, obj, drawer, location)
                        print(f"Executed {skill}({obj}, {drawer}, {location})")
                elif skill == 'execute_push':
                    # Try pushing (closing) a drawer at a location
                    if drawer and location:
                        obs, reward, done = execute_push(env, task, drawer, location)
                        print(f"Executed {skill}({drawer}, {location})")
                elif skill == 'execute_pull':
                    # Try pulling (opening) a drawer using an object (handle) at a location
                    if drawer and obj and location:
                        obs, reward, done = execute_pull(env, task, drawer, obj, location)
                        print(f"Executed {skill}({drawer}, {obj}, {location})")
                elif skill == 'execute_sweep':
                    # Try sweeping an object at a location
                    if obj and location:
                        obs, reward, done = execute_sweep(env, task, obj, location)
                        print(f"Executed {skill}({obj}, {location})")
                elif skill == 'execute_rotate':
                    # Try rotating (if implemented, else skip)
                    print(f"{skill} not implemented in this domain, skipping.")
                elif skill == 'execute_go':
                    # Try moving from one location to another
                    if len(location_names) >= 2:
                        from_loc = location_names[0]
                        to_loc = location_names[1]
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        print(f"Executed {skill}({from_loc}, {to_loc})")
                    else:
                        print(f"Not enough locations for {skill}.")
                elif skill == 'execute_gripper':
                    # Try gripper action (if implemented)
                    obs, reward, done = execute_gripper(env, task)
                    print(f"Executed {skill}()")
                else:
                    print(f"Skill {skill} not recognized.")
            except Exception as e:
                print(f"[Exploration] Exception during {skill}: {e}")

        print("[Exploration] Exploration phase complete. Check above for which skills failed due to missing predicates.")
        print("You can now analyze which predicate is missing by observing which skill(s) could not be executed due to precondition failures.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
