# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, pick up the rubbish, and place it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys expected: 'bottom_anchor_pos', 'bottom_side_pos', 'middle_anchor_pos', etc.
        # Items: 'item1', 'item2', 'item3', 'bin'
        # We'll treat 'item3' as the rubbish (since the goal says "rubbish" and only item1/2/3 are on the table)
        # We'll open the 'bottom' drawer for this plan

        # Defensive: check all required keys
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'item3', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        # Plan step mapping:
        # 1. Move to bottom_side_pos (move)
        # 2. Rotate to 90 deg (rotate)
        # 3. Move to bottom_anchor_pos (move)
        # 4. Pick the drawer handle (pick)
        # 5. Pull the drawer open (pull)
        # 6. Move away from drawer (move)
        # 7. Rotate back to 0 deg (rotate)
        # 8. Move above item3 (move)
        # 9. Pick item3 (pick)
        # 10. Move above bin (move)
        # 11. Place item3 in bin (place)

        # 1. Move to bottom_side_pos
        print("[Step 1] Move to bottom_side_pos (drawer side)")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_side_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 1!")
            return

        # 2. Rotate to 90 deg (drawer grasp orientation)
        print("[Step 2] Rotate gripper to 90 degrees (drawer grasp)")
        # We'll assume 90 deg means rotation about z axis by 90 deg
        # Get current position and build target quaternion
        # We'll use scipy.spatial.transform.Rotation to get quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        quat_90 = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        # The skill expects [x, y, z, w] order
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_90,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 2!")
            return

        # 3. Move to bottom_anchor_pos (drawer handle)
        print("[Step 3] Move to bottom_anchor_pos (drawer handle)")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_anchor_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 3!")
            return

        # 4. Pick the drawer handle (simulate pick on anchor pos)
        print("[Step 4] Pick the drawer handle (close gripper on anchor)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['bottom_anchor_pos']),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 4!")
            return

        # 5. Pull the drawer open (pull along x axis, e.g. 0.20m)
        print("[Step 5] Pull the drawer open (pull along x axis)")
        try:
            pull_distance = 0.20  # meters, adjust as needed for full open
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            return
        if done:
            print("[Task] Task ended after step 5!")
            return

        # 6. Move away from drawer (back to side pos)
        print("[Step 6] Move away from drawer (to bottom_side_pos)")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_side_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 6!")
            return

        # 7. Rotate back to 0 deg (neutral orientation)
        print("[Step 7] Rotate gripper back to 0 degrees")
        quat_0 = R.from_euler('z', 0, degrees=True).as_quat()
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_0,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 7!")
            return

        # 8. Move above item3 (rubbish)
        print("[Step 8] Move above item3 (rubbish)")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['item3']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 8!")
            return

        # 9. Pick item3 (rubbish)
        print("[Step 9] Pick item3 (rubbish)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['item3']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 9!")
            return

        # 10. Move above bin
        print("[Step 10] Move above bin")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bin']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after step 10!")
            return

        # 11. Place item3 in bin
        print("[Step 11] Place item3 in bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(positions['bin']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
