import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===
        # Step 1: Move gripper to the side position of the bottom drawer
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 1 (move to bottom_side_pos) failed: {e}")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        try:
            # Assume 90 deg is [0, 0, 0.7071, 0.7071] in xyzw (z axis)
            target_quat = np.array([0, 0, 0.7071, 0.7071])
            obs, reward, done = rotate(env, task, target_quat=target_quat)
            if done:
                print("[Task] Task ended after rotate to 90 deg!")
                return
        except Exception as e:
            print(f"[Error] Step 2 (rotate to 90 deg) failed: {e}")
            return

        # Step 3: Move gripper to anchor position of the bottom drawer
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 3 (move to bottom_anchor_pos) failed: {e}")
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        try:
            obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick (grasp) drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Step 4 (pick drawer handle) failed: {e}")
            return

        # Step 5: Pull the drawer open (along x axis, e.g., 0.15m)
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pull (open) drawer!")
                return
        except Exception as e:
            print(f"[Error] Step 5 (pull drawer) failed: {e}")
            return

        # Step 6: Move gripper away from drawer (to side position)
        try:
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after move away from drawer!")
                return
        except Exception as e:
            print(f"[Error] Step 6 (move away from drawer) failed: {e}")
            return

        # Step 7: Rotate gripper back to 0 degrees (release orientation)
        try:
            target_quat = np.array([0, 0, 0, 1])  # 0 deg
            obs, reward, done = rotate(env, task, target_quat=target_quat)
            if done:
                print("[Task] Task ended after rotate to 0 deg!")
                return
        except Exception as e:
            print(f"[Error] Step 7 (rotate to 0 deg) failed: {e}")
            return

        # Step 8: Move gripper above item1 (assume item1 is rubbish on table)
        try:
            item1_pos = positions.get('item1')
            if item1_pos is None:
                raise KeyError("item1 not found in positions.")
            # Approach from above (z axis)
            above_item1 = np.array(item1_pos) + np.array([0, 0, 0.10])
            obs, reward, done = move(env, task, target_pos=above_item1)
            if done:
                print("[Task] Task ended after move above item1!")
                return
        except Exception as e:
            print(f"[Error] Step 8 (move above item1) failed: {e}")
            return

        # Step 9: Place any held object (if holding drawer, release)
        try:
            # Place at current position (above item1)
            obs, reward, done = place(env, task, target_pos=above_item1, approach_distance=0.05, approach_axis='z')
            if done:
                print("[Task] Task ended after place (release drawer)!")
                return
        except Exception as e:
            print(f"[Error] Step 9 (place/release drawer) failed: {e}")
            return

        # Step 10: Move gripper to item1 (for pick)
        try:
            obs, reward, done = move(env, task, target_pos=item1_pos)
            if done:
                print("[Task] Task ended after move to item1!")
                return
        except Exception as e:
            print(f"[Error] Step 10 (move to item1) failed: {e}")
            return

        # Step 11: Pick item1 (rubbish) from table
        try:
            obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick item1!")
                return
        except Exception as e:
            print(f"[Error] Step 11 (pick item1) failed: {e}")
            return

        # Step 12: Move gripper above bin (for disposal)
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise KeyError("bin not found in positions.")
            above_bin = np.array(bin_pos) + np.array([0, 0, 0.10])
            obs, reward, done = move(env, task, target_pos=above_bin)
            if done:
                print("[Task] Task ended after move above bin!")
                return
        except Exception as e:
            print(f"[Error] Step 12 (move above bin) failed: {e}")
            return

        # Step 13: Place item1 into bin
        try:
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
                return
        except Exception as e:
            print(f"[Error] Step 13 (place item1 in bin) failed: {e}")
            return

        print("[Task] Task completed (all steps executed).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()