import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Pull open any drawer that is not locked, then drop the rubbish into the bin.

        # 1. Choose a drawer to open (bottom, middle, or top). We'll use 'bottom' for determinism.
        # Get relevant positions for the bottom drawer
        try:
            gripper_home_pos = positions['waypoint1'] if 'waypoint1' in positions else None
            bottom_side_pos = positions['bottom_side_pos']
            bottom_anchor_pos = positions['bottom_anchor_pos']
            bottom_joint_pos = positions['bottom_joint_pos']
            rubbish_pos = positions['rubbish']
            bin_pos = positions['bin']
        except KeyError as e:
            print(f"[Error] Missing object position: {e}")
            shutdown_environment(env)
            return

        # Step 1: Rotate gripper to 90 degrees (vertical for side approach)
        # We'll assume the gripper's orientation is quaternion, and 90deg about z is needed.
        # Get current orientation
        obs = task.get_observation()
        start_quat = obs.gripper_pose[3:7]
        # Compute target quaternion for 90deg rotation about z
        from scipy.spatial.transform import Rotation as R
        target_rot = R.from_euler('z', 90, degrees=True)
        target_quat = target_rot.as_quat()  # xyzw
        # Use the same x,y,z as current, just change orientation
        try:
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move to the side position of the bottom drawer
        try:
            obs, reward, done = move(env, task, bottom_side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move to the anchor position of the bottom drawer (for grasping handle)
        try:
            obs, reward, done = move(env, task, bottom_anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        # Use pick with approach axis 'y' (assuming handle is along y)
        try:
            obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='y')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open (pull along x axis, e.g., 0.15m)
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended during pull drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick the rubbish from the table
        # Approach from above (z axis)
        try:
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick rubbish!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place the rubbish into the bin
        # Approach from above (z axis)
        try:
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()