# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===
        # Step 1: Move gripper to the side position of an available (unlocked) drawer (choose 'bottom' drawer)
        # Step 2: Rotate gripper to 90 degrees (required for side approach)
        # Step 3: Move gripper from side to anchor position (to grab the drawer handle)
        # Step 4: Pull the drawer open
        # Step 5: Move gripper away from drawer (to a safe waypoint)
        # Step 6: Pick up rubbish from table (item3)
        # Step 7: Move to bin
        # Step 8: Place rubbish in bin

        # --- Step 1: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 1 (move to bottom_side_pos) failed: {e}")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for side approach) ---
        try:
            # 90 degrees about z axis: quaternion [x, y, z, w] for 90deg about z is [0, 0, sqrt(0.5), sqrt(0.5)]
            from scipy.spatial.transform import Rotation as R
            quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            print("[Task] Rotating gripper to 90 degrees (z axis):", quat_90deg)
            obs, reward, done = rotate(env, task, target_quat=quat_90deg)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Step 2 (rotate to 90deg) failed: {e}")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            print("[Task] Moving to bottom drawer anchor position:", bottom_anchor_pos)
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 3 (move to bottom_anchor_pos) failed: {e}")
            return

        # --- Step 4: Pull the drawer open ---
        try:
            # Pull along x axis (assume drawer opens along +x, adjust if needed)
            pull_distance = 0.18  # meters, adjust as needed for your environment
            print(f"[Task] Pulling drawer along x axis by {pull_distance} meters.")
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] Step 4 (pull drawer) failed: {e}")
            return

        # --- Step 5: Move to a safe waypoint (e.g., 'waypoint1') ---
        try:
            waypoint1 = positions.get('waypoint1')
            if waypoint1 is None:
                # If not defined, move slightly above current position
                obs_now = task.get_observation()
                curr_pos = obs_now.gripper_pose[:3]
                waypoint1 = curr_pos + np.array([0, 0, 0.10])
            print("[Task] Moving to safe waypoint after opening drawer:", waypoint1)
            obs, reward, done = move(env, task, target_pos=waypoint1)
            if done:
                print("[Task] Task ended after move to waypoint1!")
                return
        except Exception as e:
            print(f"[Error] Step 5 (move to waypoint1) failed: {e}")
            return

        # --- Step 6: Pick up rubbish (item3) from table ---
        try:
            item3_pos = positions.get('item3')
            if item3_pos is None:
                raise KeyError("item3 not found in object positions.")
            print("[Task] Picking up item3 (rubbish) from table at:", item3_pos)
            obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Step 6 (pick item3) failed: {e}")
            return

        # --- Step 7: Move to bin ---
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise KeyError("bin not found in object positions.")
            print("[Task] Moving to bin at:", bin_pos)
            obs, reward, done = move(env, task, target_pos=bin_pos)
            if done:
                print("[Task] Task ended after move to bin!")
                return
        except Exception as e:
            print(f"[Error] Step 7 (move to bin) failed: {e}")
            return

        # --- Step 8: Place rubbish in bin ---
        try:
            print("[Task] Placing item3 in bin at:", bin_pos)
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Step 8 (place item3 in bin) failed: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
