# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'item1', 'item2', 'item3', 'bin'

        # For this task, we need to:
        # 1. Open one of the drawers (bottom, middle, or top)
        # 2. Put the rubbish in the bin

        # We'll use the 'bottom' drawer for simplicity.
        # Plan:
        # 1. Move gripper to the side position of the bottom drawer
        # 2. Rotate gripper to 90 degrees (for grasping the drawer handle)
        # 3. Move gripper to the anchor position of the bottom drawer
        # 4. Pull the drawer open
        # 5. Move gripper above the rubbish (item3)
        # 6. Pick the rubbish (item3)
        # 7. Move gripper above the bin
        # 8. Place the rubbish in the bin

        # --- Step 1: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] 'bottom_side_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_side_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer grasp) ---
        # We'll assume the quaternion for 90 degrees rotation about z-axis.
        # The skill expects a quaternion (x, y, z, w).
        # Let's use scipy to get the quaternion for 90 deg about z.
        from scipy.spatial.transform import Rotation as R
        quat_90deg_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Task] Rotating gripper to 90 degrees about z-axis (drawer grasp).")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_90deg_z,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] 'bottom_anchor_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pull the drawer open ---
        # We'll pull along the x-axis by a reasonable distance (e.g., 0.15m)
        pull_distance = 0.15
        print("[Task] Pulling the drawer open along x-axis by", pull_distance)
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # --- Step 5: Move above the rubbish (item3) ---
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] 'item3' not found in object positions.")
            shutdown_environment(env)
            return

        # Approach from above: add offset in z
        approach_offset = np.array([0, 0, 0.10])
        item3_above = np.array(item3_pos) + approach_offset
        print("[Task] Moving above item3 (rubbish):", item3_above)
        obs, reward, done = move(
            env,
            task,
            target_pos=item3_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # --- Step 6: Pick the rubbish (item3) ---
        print("[Task] Picking up item3 (rubbish) at:", item3_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(item3_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick!")
            return

        # --- Step 7: Move above the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] 'bin' not found in object positions.")
            shutdown_environment(env)
            return

        bin_above = np.array(bin_pos) + approach_offset
        print("[Task] Moving above bin:", bin_above)
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above bin!")
            return

        # --- Step 8: Place the rubbish in the bin ---
        print("[Task] Placing item3 (rubbish) in bin at:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(bin_pos),
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
