# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock a cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # 1. Find an accessible (unlocked) drawer. In the initial state, 'bottom' is not locked, 'middle' and 'top' are locked.
        # We'll open the 'bottom' drawer.

        # Get relevant positions for the bottom drawer
        # Convention: anchor_pos = handle, side_pos = approach, joint_pos = drawer center
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
            bottom_side_pos = positions['bottom_side_pos']
            bottom_joint_pos = positions['bottom_joint_pos']
        except KeyError as e:
            print(f"[Error] Missing position for: {e}")
            shutdown_environment(env)
            return

        # 1. [move] Move gripper to the side position of the bottom drawer (approach)
        print("[Task] Moving to side position of bottom drawer.")
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move to side position!")
            return

        # 2. [rotate] Rotate gripper to 90 degrees (to align with handle)
        # We'll assume that the quaternion for 90 degrees about z is known or can be constructed.
        # If not, use the current position and rotate about z.
        from scipy.spatial.transform import Rotation as R
        def get_quat_for_ninety_deg_z():
            # 90 deg about z axis
            rot = R.from_euler('z', 90, degrees=True)
            return rot.as_quat()  # xyzw
        ninety_deg_quat = get_quat_for_ninety_deg_z()
        print("[Task] Rotating gripper to 90 degrees about z.")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=ninety_deg_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during rotate!")
            return

        # 3. [pick] Move to anchor (handle) and grasp the drawer handle
        print("[Task] Picking (grasping) the bottom drawer handle.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick (drawer handle)!")
            return

        # 4. [pull] Pull the drawer open along the x axis (assume positive x is outwards)
        # We'll use a reasonable pull distance (e.g., 0.15m)
        print("[Task] Pulling the bottom drawer open.")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pull (drawer)!")
            return

        # 5. [place] Release the drawer handle (optional, but for completeness)
        # We'll place at the current position (handle), just open the gripper
        print("[Task] Releasing the drawer handle.")
        obs, reward, done = place(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.05,
            max_steps=50,
            threshold=0.01,
            approach_axis='z',
            timeout=5.0
        )
        if done:
            print("[Task] Task ended during place (release handle)!")
            return

        # === Now, throw away the trash ===
        # Find the trash object and bin position
        # The object list says 'item1', 'item2', 'item3', 'bin'
        # We'll assume 'item3' is the trash (since the goal says "throw away the trash" and leave other objects alone)
        try:
            item3_pos = positions['item3']
            bin_pos = positions['bin']
        except KeyError as e:
            print(f"[Error] Missing position for: {e}")
            shutdown_environment(env)
            return

        # 6. [move] Move to above the trash (item3)
        print("[Task] Moving to trash (item3).")
        obs, reward, done = move(
            env,
            task,
            target_pos=item3_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move to trash!")
            return

        # 7. [pick] Pick up the trash (item3)
        print("[Task] Picking up the trash (item3).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item3_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick (trash)!")
            return

        # 8. [move] Move to above the bin
        print("[Task] Moving to bin.")
        obs, reward, done = move(
            env,
            task,
            target_pos=bin_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended during move to bin!")
            return

        # 9. [place] Place the trash in the bin
        print("[Task] Placing the trash in the bin.")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Exception] {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
