import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, pick up the rubbish, and place it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys expected: 'rubbish', 'bin', 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Plan: Open a drawer fully, pick up rubbish, place in bin ---

        # 1. Select a drawer to open (choose 'bottom' drawer for simplicity)
        #    We'll use 'bottom_side_pos' (side of bottom drawer), 'bottom_anchor_pos' (handle), etc.
        #    The gripper starts at 'nowhere_pos' and needs to rotate, move to side, move to anchor, pick-drawer, pull.

        # Step 1: Rotate gripper to 90 degrees (vertical for grasping drawer handle)
        # We'll use a quaternion for 90 deg rotation about z axis
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        # RLBench/pyrep expects [x, y, z, w] order
        # skill_code expects [x, y, z, w]
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            return

        # Step 2: Move to the side position of the bottom drawer
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side position: {e}")
            return

        # Step 3: Move to the anchor (handle) position of the bottom drawer
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor position: {e}")
            return

        # Step 4: (Redundant move, but included for plan completeness) - stay at anchor
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during redundant move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during redundant move to anchor position: {e}")
            return

        # Step 5: Pull the drawer open (pull along x axis, e.g., 0.18m)
        try:
            # Estimate pull distance (positive x direction, typical for drawers)
            pull_distance = 0.18
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            return

        # Step 6: Pick up the rubbish from the table (assume rubbish is now accessible)
        try:
            rubbish_pos = positions.get('rubbish')
            if rubbish_pos is None:
                raise KeyError("rubbish not found in object positions.")
            obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")
            return

        # Step 7: Place the rubbish in the bin
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise KeyError("bin not found in object positions.")
            obs, reward, done = place(
                env,
                task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Task] Exception during place: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()