import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===
        # Step 1: Rotate gripper to 90 degrees (vertical) for drawer manipulation
        # Step 2: Move to the side position of the bottom drawer
        # Step 3: Move to the anchor position of the bottom drawer
        # Step 4: Pick (grasp) the drawer handle at anchor position
        # Step 5: Pull the drawer open
        # Step 6: Pick up the rubbish from the table
        # Step 7: Place the rubbish in the bin

        # --- Step 1: Rotate gripper to 90 degrees (vertical) ---
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Target orientation: 90 degrees about x axis (vertical)
            # We'll use scipy to get the quaternion for 90 deg about x
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('x', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to the side position of the bottom drawer ---
        try:
            # Use 'bottom_side_pos' as the side position for the bottom drawer
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise ValueError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended after move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to the anchor position of the bottom drawer ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        try:
            # For picking the drawer, approach along -z (downward) axis
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='-z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        try:
            # Pull along x axis (assume drawer opens along +x, adjust if needed)
            pull_distance = 0.18  # 18cm typical drawer pull
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull drawer) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up the rubbish from the table ---
        try:
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise ValueError("rubbish not found in object positions.")
            # Approach from above (z axis)
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick rubbish) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise ValueError("bin not found in object positions.")
            # Place from above (z axis)
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Step 7 (place rubbish) failed: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()