import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock the cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys expected in positions:
        # 'rubbish', 'tomato1', 'tomato2', 'bin', 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Step 1: Rotate gripper to 90 degrees (for side approach to drawer) ---
        # We'll use the bottom drawer (unlocked) for accessibility.
        # Assume 'ninety_deg' means gripper z-axis is horizontal (e.g., [0, 0, 0.7071, 0.7071])
        # If object_positions provides a quaternion for this, use it; else, construct it.
        try:
            # Try to get a predefined quaternion for 90 degrees if available
            target_quat = positions.get('ninety_deg_quat', None)
            if target_quat is None:
                # 90 deg about y axis: [x, y, z, w] = [0, 0.7071, 0, 0.7071]
                # But RLBench uses [x, y, z, w] order
                target_quat = np.array([0, 0, 0.7071, 0.7071])
        except Exception as e:
            print("[Task] Could not get 90 deg quaternion from positions, using default.")
            target_quat = np.array([0, 0, 0.7071, 0.7071])

        print("[Task] Step 1: Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 2: Move to side position of bottom drawer ---
        # Use 'bottom_side_pos' as the approach position for the gripper
        try:
            side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Task] Error: 'bottom_side_pos' not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Step 2: Moving to bottom drawer side position.")
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to side position!")
            return

        # --- Step 3: Move to anchor position of bottom drawer (for grasping handle) ---
        try:
            anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Task] Error: 'bottom_anchor_pos' not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Step 3: Moving to bottom drawer anchor position.")
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        # Use pick with approach axis '-y' (assuming drawer handle is along y axis)
        print("[Task] Step 4: Grasping the bottom drawer handle.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='-y',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along the x axis (assume positive x is outwards from cabinet)
        print("[Task] Step 5: Pulling the bottom drawer open.")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.18,  # Adjust as needed for full open
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (open drawer)!")
            return

        # --- Step 6: Pick up the rubbish from the table ---
        try:
            rubbish_pos = positions['rubbish']
        except KeyError:
            print("[Task] Error: 'rubbish' not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Step 6: Picking up the rubbish.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (rubbish)!")
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Task] Error: 'bin' not found in positions.")
            shutdown_environment(env)
            return

        print("[Task] Step 7: Placing the rubbish in the bin.")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()