# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- PLAN EXECUTION LOGIC ---
        # Goal: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.

        # 1. Choose a drawer to open (bottom drawer is unlocked and available)
        drawer_name = 'bottom'
        gripper_name = 'gripper'
        # Positions for the bottom drawer
        side_pos_key = 'bottom_side_pos'
        anchor_pos_key = 'bottom_anchor_pos'
        # For the plan, we use the following keys for positions
        # These should be present in positions dict
        try:
            side_pos = positions[side_pos_key]
            anchor_pos = positions[anchor_pos_key]
        except KeyError as e:
            print(f"[Error] Missing position for {e}. Check object_positions module.")
            shutdown_environment(env)
            return

        # 1. [rotate] Rotate gripper to 90 degrees (vertical for side approach)
        # Assume 90 deg quaternion is provided in positions or can be constructed
        # If not, use a default quaternion for 90 deg about z axis
        try:
            # Try to get a precomputed quaternion for 90 deg if available
            ninety_deg_quat = positions.get('ninety_deg_quat', None)
            if ninety_deg_quat is None:
                # Default: 90 deg about z axis
                from scipy.spatial.transform import Rotation as R
                ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        except Exception as e:
            print(f"[Error] Could not compute 90 deg quaternion: {e}")
            shutdown_environment(env)
            return

        print("[Task] Step 1: Rotating gripper to 90 degrees (vertical)")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=ninety_deg_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # 2. [move] Move gripper to the side position of the bottom drawer
        print(f"[Task] Step 2: Moving gripper to side position of {drawer_name} drawer at {side_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to side position!")
            return

        # 3. [move] Move gripper from side position to anchor position (for pulling)
        print(f"[Task] Step 3: Moving gripper to anchor position of {drawer_name} drawer at {anchor_pos}")
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # 4. [pick] Grasp the drawer handle at anchor position
        print(f"[Task] Step 4: Picking (grasping) the {drawer_name} drawer handle at anchor position")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # 5. [pull] Pull the drawer open along the x axis (assume positive x is outwards)
        # Estimate a reasonable pull distance (e.g., 0.15m)
        pull_distance = 0.15
        print(f"[Task] Step 5: Pulling the {drawer_name} drawer open by {pull_distance}m along x axis")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # 6. [pick] Pick up the rubbish from the table
        # Find rubbish position
        try:
            rubbish_pos = positions['rubbish']
        except KeyError:
            print("[Error] Rubbish object position not found!")
            shutdown_environment(env)
            return
        print(f"[Task] Step 6: Picking up rubbish at {rubbish_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (rubbish)!")
            return

        # 7. [place] Place the rubbish into the bin
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] Bin object position not found!")
            shutdown_environment(env)
            return
        print(f"[Task] Step 7: Placing rubbish into bin at {bin_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
