import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Task Plan Execution ===
        # Goal: Pull open any drawer that is not locked, then drop the rubbish into the bin.

        # 1. Choose a drawer to open (bottom, middle, or top). We'll try bottom first.
        # 2. Move gripper to the side position of the drawer.
        # 3. Rotate gripper to 90 degrees (for side grasp).
        # 4. Move to anchor position.
        # 5. Pick the drawer handle (anchor).
        # 6. Pull the drawer open.
        # 7. Move gripper above the rubbish (item3).
        # 8. Pick the rubbish.
        # 9. Move above the bin.
        # 10. Place the rubbish in the bin.

        # --- Step 1: Move to bottom_side_pos ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in object positions.")
            shutdown_environment(env)
            return

        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (xyzw for 90deg about z) ---
        from scipy.spatial.transform import Rotation as R
        quat_90deg_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw

        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=quat_90deg_z)
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after rotate to 90deg!")
            return

        # --- Step 3: Move to bottom_anchor_pos ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in object positions.")
            shutdown_environment(env)
            return

        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick the drawer handle (anchor) ---
        # Approach axis: 'y' or 'z' depending on drawer geometry; we'll use 'y' as a guess.
        # [Frozen Code Start]
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='y')
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after pick drawer handle!")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along x axis (assume drawers open along +x), distance ~0.15m
        # [Frozen Code Start]
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after pull!")
            return

        # --- Step 6: Move above the rubbish (item3) ---
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] item3 not found in object positions.")
            shutdown_environment(env)
            return

        # Move above item3 (add offset in z for approach)
        item3_above = np.array(item3_pos) + np.array([0, 0, 0.10])
        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=item3_above)
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # --- Step 7: Pick the rubbish (item3) ---
        obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after pick item3!")
            return

        # --- Step 8: Move above the bin ---
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] bin not found in object positions.")
            shutdown_environment(env)
            return

        bin_above = np.array(bin_pos) + np.array([0, 0, 0.10])
        obs, reward, done = move(env, task, target_pos=bin_above)
        if done:
            print("[Task] Task ended after move above bin!")
            return

        # --- Step 9: Place the rubbish in the bin ---
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()