import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Map object names to variables for clarity
        # Drawer positions
        bottom_anchor_pos = positions.get('bottom_anchor_pos')
        bottom_joint_pos = positions.get('bottom_joint_pos')
        bottom_side_pos = positions.get('bottom_side_pos')
        middle_anchor_pos = positions.get('middle_anchor_pos')
        middle_joint_pos = positions.get('middle_joint_pos')
        middle_side_pos = positions.get('middle_side_pos')
        top_anchor_pos = positions.get('top_anchor_pos')
        top_joint_pos = positions.get('top_joint_pos')
        top_side_pos = positions.get('top_side_pos')
        # Items
        item1_pos = positions.get('item1')
        item2_pos = positions.get('item2')
        item3_pos = positions.get('item3')  # item3 is rubbish
        bin_pos = positions.get('bin')

        # === Oracle Plan Execution ===
        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=quat_90_x)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='y')
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        obs, reward, done = move(env, task, target_pos=item3_pos)
        obs, reward, done = pick(env, task, target_pos=item3_pos, approach_distance=0.10, approach_axis='-z')
        # [Frozen Code End]

        if done:
            print("[Task] Task ended after picking up the rubbish!")
            shutdown_environment(env)
            print("===== End of Skeleton Task =====")
            return

        # Step: Move to bin position
        print("[Step] Move to bin position")
        try:
            obs, reward, done = move(env, task, target_pos=bin_pos)
        except Exception as e:
            print(f"[Error] Exception during move to bin: {e}")
            shutdown_environment(env)
            print("===== End of Skeleton Task =====")
            return
        if done:
            print("[Task] Task ended after move to bin!")
            shutdown_environment(env)
            print("===== End of Skeleton Task =====")
            return

        # Step: Place rubbish in bin
        print("[Step] Place rubbish in bin")
        try:
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.10, approach_axis='-z')
        except Exception as e:
            print(f"[Error] Exception during place in bin: {e}")
            shutdown_environment(env)
            print("===== End of Skeleton Task =====")
            return
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()