import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then drop the rubbish into the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'bin'
        # ]
        # We'll treat 'item3' as the rubbish (since the goal says "rubbish" and the object list has item1, item2, item3).
        # We'll open the "bottom" drawer (bottom_anchor_pos, bottom_joint_pos, bottom_side_pos).

        # Defensive: check all required keys exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'item3', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        # Step 1: Move to the side position of the bottom drawer
        print("[Plan] Step 1: Move to bottom_side_pos (side of bottom drawer)")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_side_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        print("[Plan] Step 2: Rotate gripper to 90 degrees")
        # We'll assume the gripper's orientation is a quaternion, and that
        # positions['bottom_side_pos'] is the correct place to rotate.
        # Let's get the quaternion for 90 degrees about z axis.
        from scipy.spatial.transform import Rotation as R
        quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        # RLBench/pyrep expects [x, y, z, w] order
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_90deg,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position of the bottom drawer (for grasping handle)
        print("[Plan] Step 3: Move to bottom_anchor_pos (anchor of bottom drawer)")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_anchor_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Step 4: Pick the drawer handle (simulate grasp)
        print("[Plan] Step 4: Pick (close gripper) at anchor position to grasp drawer handle")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['bottom_anchor_pos']),
            approach_distance=0.05,  # Small approach, since already at anchor
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # Step 5: Pull the drawer open (pull along x axis, e.g., 0.15m)
        print("[Plan] Step 5: Pull the drawer open along x axis")
        pull_distance = 0.15  # meters, adjust as needed for your environment
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 6: Move to the rubbish (item3) on the table
        print("[Plan] Step 6: Move to item3 (rubbish) on the table")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['item3']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item3!")
            return

        # Step 7: Pick the rubbish (item3)
        print("[Plan] Step 7: Pick item3 (rubbish)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['item3']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (item3)!")
            return

        # Step 8: Move to the bin
        print("[Plan] Step 8: Move to bin")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bin']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bin!")
            return

        # Step 9: Place the rubbish in the bin
        print("[Plan] Step 9: Place item3 (rubbish) in bin")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(positions['bin']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()