# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Extract required positions for the plan
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
            bottom_joint_pos = positions['bottom_joint_pos']
            rubbish_pos = positions['rubbish']
            bin_pos = positions['bin']
        except KeyError as e:
            print(f"[Error] Required object position missing: {e}")
            shutdown_environment(env)
            return

        # Set pull distance and axis for opening the drawer
        pull_distance = 0.18  # This value may need adjustment for your environment
        pull_axis = 'x'       # Assuming drawer opens along x-axis

        # === Oracle Plan Execution ===

        # [Frozen Code Start]
        obs, reward, done = move(env, task, bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_joint_pos)
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        obs, reward, done = pick(
            env,
            task,
            target_pos=rubbish_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        # [Frozen Code End]

        if done:
            print("[Task] Task ended after pick!")
            shutdown_environment(env)
            return

        # Place the rubbish in the bin
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Exception] {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()