# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===
        # Goal: Open one of the drawers in the cabinet, then put the rubbish in the bin.

        # We'll choose the "bottom" drawer for this plan.
        # The relevant objects/positions for the bottom drawer:
        # - bottom_side_pos: side approach position for the bottom drawer
        # - bottom_anchor_pos: anchor position for grasping the bottom drawer handle
        # - bottom_joint_pos: (optional, not used directly here)
        # - rubbish: the object to pick and place in the bin
        # - bin: the target location for rubbish

        # Step 1: Rotate gripper to 90 degrees (vertical) for drawer manipulation
        try:
            # Get current gripper pose and compute target quaternion for 90 deg rotation about z
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # 90 deg about z axis in quaternion (xyzw)
            from scipy.spatial.transform import Rotation as R
            rot_90 = R.from_euler('z', 90, degrees=True)
            target_quat = rot_90.as_quat()
            # Use the same x, y, z as current, just change orientation
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move to the side position of the bottom drawer
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise ValueError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, bottom_side_pos)
            if done:
                print("[Task] Task ended during move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to bottom_side_pos: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move to the anchor position of the bottom drawer (for grasping handle)
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, bottom_anchor_pos)
            if done:
                print("[Task] Task ended during move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to bottom_anchor_pos: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        try:
            # For drawer, we use pick with the anchor position
            obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during pick (drawer handle)!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick (drawer handle): {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open (along x axis, positive direction)
        try:
            # Estimate a reasonable pull distance (e.g., 0.18m)
            pull_distance = 0.18
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended during pull (drawer)!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pull (drawer): {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick the rubbish from the table
        try:
            rubbish_pos = positions.get('rubbish')
            if rubbish_pos is None:
                raise ValueError("rubbish not found in object positions.")
            # Approach from above (z axis)
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick (rubbish)!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick (rubbish): {e}")
            shutdown_environment(env)
            return

        # Step 7: Place the rubbish in the bin
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise ValueError("bin not found in object positions.")
            # Approach from above (z axis)
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Exception during place (rubbish in bin): {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
